# AlgeriAPIs - Access Algerian Data via Public APIs
# Version 0.1.0
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.


#' Get Country Information for Algeria
#'
#' @description
#' Retrieves comprehensive country information for Algeria from the REST Countries API.
#' This function fetches data including official and common names, geographical information,
#' capital, area, population, and languages.
#'
#' @return A tibble with one row containing Algeria's country information, or NULL if the API is unavailable.
#'
#' @examples
#' \donttest{
#' dz_info <- get_country_info_dz()
#' print(dz_info)
#' }
#'
#' @importFrom httr GET http_error content
#' @importFrom jsonlite fromJSON
#' @importFrom tibble tibble
#'
#' @export
get_country_info_dz <- function() {
  url <- "https://restcountries.com/v3.1/name/algeria?fullText=true"
  # Intentar obtener los datos y capturar errores de conexión
  response <- tryCatch(
    httr::GET(url),
    error = function(e) {
      message("Could not connect to restcountries.com: ", e$message)
      return(NULL)
    }
  )
  if (is.null(response)) return(NULL)  # Si la conexión falla, retornar NULL
  # Verificar código HTTP
  if (httr::http_error(response)) {
    message("API request failed with status code: ", httr::status_code(response))
    return(NULL)
  }
  # Convertir a texto y parsear JSON
  data_raw <- httr::content(response, as = "text", encoding = "UTF-8")
  data_list <- tryCatch(
    jsonlite::fromJSON(data_raw),
    error = function(e) {
      message("Error parsing JSON: ", e$message)
      return(NULL)
    }
  )
  if (is.null(data_list) || length(data_list) == 0) {
    message("No data found for Algeria.")
    return(NULL)
  }
  # Tomar solo el primer país
  data <- data_list[1, ]
  tibble::tibble(
    name_common   = data$name$common,
    name_official = data$name$official,
    region        = data$region,
    subregion     = data$subregion,
    capital       = paste(data$capital, collapse = ", "),
    area          = data$area,
    population    = data$population,
    languages     = paste(unlist(data$languages), collapse = ", ")
  )
}
