% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdCrossprod_hdf5}
\alias{bdCrossprod_hdf5}
\title{Crossprod with hdf5 matrix}
\usage{
bdCrossprod_hdf5(
  filename,
  group,
  A,
  B = NULL,
  groupB = NULL,
  block_size = NULL,
  mixblock_size = NULL,
  paral = NULL,
  threads = NULL,
  outgroup = NULL,
  outdataset = NULL,
  overwrite = NULL
)
}
\arguments{
\item{filename}{String indicating the HDF5 file path}

\item{group}{String indicating the input group containing matrix A}

\item{A}{String specifying the dataset name for matrix A}

\item{B}{Optional string specifying dataset name for matrix B.
If NULL, performs A^t * A}

\item{groupB}{Optional string indicating group containing matrix B.
If NULL, uses same group as A}

\item{block_size}{Optional integer specifying the block size for processing.
Default is automatically determined based on matrix dimensions}

\item{mixblock_size}{Optional integer for memory block size in parallel processing}

\item{paral}{Optional boolean indicating whether to use parallel processing.
Default is false}

\item{threads}{Optional integer specifying number of threads for parallel processing.
If NULL, uses maximum available threads}

\item{outgroup}{Optional string specifying output group.
Default is "OUTPUT"}

\item{outdataset}{Optional string specifying output dataset name.
Default is "CrossProd_A_x_B"}

\item{overwrite}{Optional boolean indicating whether to overwrite existing datasets.
Default is false}
}
\value{
A list containing the location of the crossproduct result:
\describe{
\item{fn}{Character string. Path to the HDF5 file containing the result}
\item{ds}{Character string. Full dataset path to the crossproduct
result (t(A) \%\emph{\% A or t(A) \%}\% B) within the HDF5 file}
}
}
\description{
Performs optimized cross product operations on matrices stored in HDF5 format.
For a single matrix A, computes A^t * A. For two matrices A and B, computes
A^t * B. Uses block-wise processing for memory efficiency.
}
\details{
The function implements block-wise matrix multiplication to handle large matrices
efficiently. Block size is automatically optimized based on:
\itemize{
\item Available memory
\item Matrix dimensions
\item Whether parallel processing is enabled
}

For parallel processing:
\itemize{
\item Uses OpenMP for thread management
\item Implements cache-friendly block operations
\item Provides automatic thread count optimization
}

Memory efficiency is achieved through:
\itemize{
\item Block-wise reading and writing
\item Minimal temporary storage
\item Proper resource cleanup
}
}
\examples{
\dontrun{
  library(BigDataStatMeth)
  library(rhdf5)
  
  # Create test matrix
  N = 1000
  M = 1000
  set.seed(555)
  a <- matrix(rnorm(N*M), N, M)
  
  # Save to HDF5
  bdCreate_hdf5_matrix("test.hdf5", a, "INPUT", "A", overwriteFile = TRUE)
  
  # Compute cross product
  bdCrossprod_hdf5("test.hdf5", "INPUT", "A", 
                   outgroup = "OUTPUT",
                   outdataset = "result",
                   block_size = 1024,
                   paral = TRUE,
                   threads = 4)
}

}
