% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simBuyseTest.R
\name{simBuyseTest}
\alias{simBuyseTest}
\title{Simulation of data for the BuyseTest}
\usage{
simBuyseTest(
  n.T,
  n.C = NULL,
  argsBin = list(),
  argsCont = list(),
  argsTTE = list(),
  names.strata = NULL,
  level.strata = NULL,
  n.strata = NULL,
  name.cluster = "id",
  prefix.cluster = NULL,
  name.treatment = "treatment",
  level.treatment = c("C", "T"),
  format = "data.table",
  latent = FALSE
)
}
\arguments{
\item{n.T}{[integer, >0] number of patients in the treatment arm}

\item{n.C}{[integer, >0] number of patients in the control arm}

\item{argsBin}{[list] arguments to be passed to \code{simBuyseTest_bin}. They specify the distribution parameters of the categorical endpoints.}

\item{argsCont}{[list] arguments to be passed to \code{simBuyseTest_continuous}. They specify the distribution parameters of the continuous endpoints.}

\item{argsTTE}{[list]  arguments to be passed to \code{simBuyseTest_TTE}. They specify the distribution parameters of the time to event endpoints.}

\item{names.strata}{[character vector] name of the strata variables. Must have same length as \code{n.strata}.}

\item{level.strata}{[list of character vector] value associated to each strata. Must have same length as \code{n.strata}.}

\item{n.strata}{[integer, >0] number of strata. \code{NULL} indicates no strata.}

\item{name.cluster}{[character] name of the cluster variable. If \code{NULL} no cluster variable is created.}

\item{prefix.cluster}{[character] character string to be added to the cluster index.}

\item{name.treatment}{[character] name of the treatment variable.}

\item{level.treatment}{[character vector of length 2] levels of the treatment variable.}

\item{format}{[character] the format of the output. Can be \code{"data.table"}, \code{"data.frame"}, \code{"matrix"} or \code{"function"}.}

\item{latent}{[logical] If \code{TRUE} also export the latent variables (e.g. censoring times or event times).}
}
\value{
Depends on the argument \code{format}: either a data.frame, data.table, matrix containing the simulated data, 
or a function that can be used to simulate data.
}
\description{
Simulate categorical, continuous or time to event endpoints, possibly along with a strata variable.
Categorical endpoints are simulated by thresholding a latent Gaussian variable (tobit model),
continuous endpoints are simulated using a Gaussian distribution,
and time to event endpoints are simulated using Weibull distributions for the event of interest, competing events, and censoring.
This function is built upon the \code{lvm} and \code{sim} functions from the lava package.
}
\details{
Endpoints are simulated independently of the strata variable and independently of each other,
with the exception of categorical endpoint and the time to event endpoints that can be correlated
by specifying a non-0 value for the \code{rho.T} and \code{rho.C} elements of the argument \code{argsBin}.

Arguments in the list \code{argsBin}:
\itemize{
    \item\code{p.T} list of probabilities for the values taken by each endpoint (categorical endpoint, treatment group). 
    \item\code{p.C} same as \code{p.T} but for the control group. 
    \item\code{rho.T} value of the regression coefficient between the underlying latent variable and the survival time.
    Only implemented for weibull distributed survival times.
    \item\code{rho.C} same as \code{rho.T} but for the control group. 
    \item\code{name} names of the binary variables.
}

Arguments in the list \code{argsCont}:
    \itemize{
    \item\code{mu.T} expected value of each endpoint (continuous endpoint, treatment group). 
    \item\code{mu.C} same as \code{mu.C} but for the control group. 
    \item\code{sigma.T} standard deviation of the values of each endpoint (continuous endpoint, treatment group). 
    \item\code{sigma.C} same as \code{sigma.T} but for the control group. 
    \item\code{name} names of the continuous variables.
    }

Arguments in the list \code{argsTTE}:
    \itemize{
    \item\code{CR} should competing risks be simulated? 
    \item\code{scale.T,scale.C,scale.CR,scale.censoring.T,scale.censoring.C} scale parameter of the Weibull distribution for, respectively,
     the event of interest in the treatment group,
     the event of interest in the control group,
     the competing event in both groups,
     the censoring mechanism in the treatment group,
     the censoring mechanism in the control group
    \item\code{shape.T,shape.C,shape.CR,shape.censoring.T,shape.censoring.C} shape parameter of the Weibull distribution for, respectively,
     the event of interest in the treatment group,
     the event of interest in the control group,
     the competing event in both groups,
     the censoring mechanism in the treatment group,
     the censoring mechanism in the control group
    \item\code{dist.T,dist.C,dist.CR,dist.censoring.T,dist.censoring.C} type of distribution (\code{"weibull"}, \code{"uniform"}, \code{"piecewiseExp"}) for, respectively,
     the event of interest in the treatment group,
     the event of interest in the control group,
     the competing event in both groups,
     the censoring mechanism in the treatment group,
     the censoring mechanism in the control group.
     For uniform distirbutions the (scale,shape) parameters becomes the support (min, max) of the censoring distribution.
     For piecewise exponential distributions the (scale,shape) should be lists of numeric (see example)
     and the shape parameters becomes the time parameters (first element should be 0).
    \item\code{name} names of the time to event variables. 
    \item\code{name.censoring} names of the event type indicators. #'      
    }
}
\examples{
library(data.table)

n <- 1e2

#### by default ####
simBuyseTest(n)

## with a strata variable having 5 levels
simBuyseTest(n, n.strata = 5)
## with a strata variable named grade
simBuyseTest(n, n.strata = 5, names.strata = "grade")
## several strata variables
simBuyseTest(1e3, n.strata = c(2,4), names.strata = c("Gender","AgeCategory"))

#### only categorical endpoints ####
args <- list(p.T = list(c(low=0.1,moderate=0.5,high=0.4)))
dt.bin <- simBuyseTest(n, argsBin = args, argsCont = NULL, argsTTE = NULL)
table(dt.bin$toxicity)/NROW(dt.bin)

args <- list(p.T = list(c(low=0.1,moderate=0.5,high=0.4), c(0.1,0.9)))
dt.bin <- simBuyseTest(n, argsBin = args, argsCont = NULL, argsTTE = NULL)
table(dt.bin$toxicity1)/NROW(dt.bin)
table(dt.bin$toxicity2)/NROW(dt.bin)

#### only continuous endpoints ####
args <- list(mu.T = c(3:5/10), sigma.T = rep(1,3))
dt.cont <- simBuyseTest(n, argsBin = NULL, argsCont = args, argsTTE = NULL)
c(mean(dt.cont$score1), mean(dt.cont$score2), mean(dt.cont$score3))
c(sd(dt.cont$score1), sd(dt.cont$score2), sd(dt.cont$score3))

#### only TTE endpoints ####
## weibull distributed
args <- list(scale.T = c(3:5/10), scale.censoring.T = rep(1,3))
dt.tte <- simBuyseTest(n, argsBin = NULL, argsCont = NULL, argsTTE = args)
1/c(sum(dt.tte$eventtime1)/sum(dt.tte$status1),
  sum(dt.tte$eventtime2)/sum(dt.tte$status2),
  sum(dt.tte$eventtime3)/sum(dt.tte$status3))
        
1/c(sum(dt.tte$eventtime1)/sum(dt.tte$status1==0),
  sum(dt.tte$eventtime2)/sum(dt.tte$status2==0),
  sum(dt.tte$eventtime3)/sum(dt.tte$status3==0))

hist(dt.tte$eventtime1)

## uniform distributed
args <- list(scale.T = 0, shape.T = 1, dist.T = "uniform", scale.censoring.T = 1e5,
             scale.C = 0, shape.C = 2, dist.C = "uniform", scale.censoring.C = 1e5)
dt.tte <- simBuyseTest(n, argsBin = NULL, argsCont = NULL, argsTTE = args)

par(mfrow=c(1,2))
hist(dt.tte$eventtime[dt.tte$treatment=="C"])
hist(dt.tte$eventtime[dt.tte$treatment=="T"])

## piecewise constant exponential distributed
## time [0;4]: scale parameter 10
## time [4;12]: scale parameter 13
## time [12;18.]: scale parameter 18
## time [18.5;36]: scale parameter 31
## after that: scale parameter 37
vec.scale <- list(c(10,13,18,31,100))
vec.time <- list(c(0,4,12,18.5,36))
args <- list(scale.T = vec.scale, shape.T = vec.time, dist.T = "piecewiseExp",
             scale.C = 10, shape.C = 1, dist.C = "weibull",
             scale.censoring.T = 1e5)
dt.tte <- simBuyseTest(n, argsBin = NULL, argsCont = NULL, argsTTE = args)

if(require(prodlim)){
plot(prodlim(Hist(eventtime,status)~treatment, data = dt.tte))
}

#### correlated categorical / time to event endpoint ####
## WARNING: only for weibull distributed time to event endpoint
args.bin <- list(p.T = list(c(low=0.1,moderate=0.5,high=0.4)), rho.T = 1)
args.tte <- list(scale.T = 2, scale.censoring.T = 1)
dt.corr <- simBuyseTest(n, argsBin = args.bin, argsCont = NULL, argsTTE = args.tte)

1/(sum(dt.corr$eventtime)/sum(dt.corr$status))
1/(sum(dt.corr$eventtime)/sum(dt.corr$status==0))
table(dt.corr$toxicity)/NROW(dt.corr)

boxplot(eventtime ~ toxicity, data = dt.corr)

#### pre-computation ####
library(lava)
mySimulator <- simBuyseTest(n, format = "function") ## creates lvm object once for all
set.seed(1)
sim(mySimulator)
set.seed(2)
sim(mySimulator) 

}
\author{
Brice Ozenne
}
\keyword{datagen}
