#' plotScm
#'
#' This function is used to generate the sample by component matrix plot.
#' @param scm Sample by component matrix
#' @param runName Name of the run used in plot titles, Default is ""
#' @param saveRes Whether or not to save the plots, Default is FALSE
#' @param saveDir Where to save the plots, must be provided if using saveDir
#' @param rowOrder Ordering specification for the rows of the heatmap. Three possible options:
#' * TRUE:  Uses hierarchical clustering to determine row order.
#' * FALSE: (default) Leaves rows in the order they were given.
#' * A numeric vector the same length as the number of rows specifying the indices of the input matrix
#' @param colOrder Ordering specification for the columns of the heatmap. See above for options. Default value is T.
#' @return pheatmap figure of component result by sample
#' @keywords plot scm matrix
#' @export
#' @examples
#' plotScm(scmExp)
#' 
#' plotScm(scmExp, rowOrder = FALSE, colOrder = FALSE)
#' 
#' newOrder = sample(1:ncol(scmExp), ncol(scmExp))
#' plotScm(scmExp, colOrder = newOrder)
plotScm = function(scm, runName="", saveRes = FALSE, saveDir = NULL, rowOrder = FALSE, colOrder = TRUE) {

  #Checks the rowOrder parameter
  if (is.numeric(rowOrder)){
    if (length(rowOrder) != nrow(scm)){
      msg = "rowOrder in plotScm call does not have the correct amount of values."
      msg = paste0(msg,"\n Found ",length(rowOrder)," values, expected ",nrow(scm),".")
      msg = paste0(msg,"\n Defaulting to heirarchical clustering for row order.")
      warning(msg)
      rowOrder = TRUE
    } else {
      scm = scm[rowOrder, ]
      rowOrder = FALSE
    }
  } else if (!is.logical(rowOrder)){
    msg = "rowOrder in plotScm call in unrecognized format."
    msg = paste0(msg,"\n Please see documentation using ?plotScm.")
    msg = paste0(msg,"\n Defaulting to heirarchical clustering for row order.")
    warning(msg)
    rowOrder = TRUE
  }
  
  #Checks the colOrder parameter
  if (is.numeric(colOrder)){
    if (length(colOrder) != ncol(scm)){
      msg = "colOrder in plotScm call does not have the correct amount of values."
      msg = paste0(msg,"\n Found ",length(colOrder)," values, expected ",ncol(scm),".")
      msg = paste0(msg,"\n Defaulting to heirarchical clustering for column order.")
      warning(msg)
      colOrder = TRUE
    } else {
      scm = scm[,colOrder]
      colOrder = FALSE
    }
  } else if (!is.logical(colOrder)){
    msg = "colOrder in plotScm call in unrecognized format."
    msg = paste0(msg,"\n Please see documentation using ?plotScm.")
    msg = paste0(msg,"\n Defaulting to heirarchical clustering for column order.")
    warning(msg)
    colOrder = TRUE
  }
  
  #Creats plotTitle based on the whether the runName is specified
  if (nchar(runName) > 0){
    plotTitle = paste0("Sample by Component matrix (",runName,")")
  } else {
    plotTitle = "Sample by Component matrix"
  }
  
  
  if (saveRes){
    if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
    }
    grDevices::tiff(paste0(saveDir,"/Sample_component_heatMap",".tiff"),width = 6,
                    height = 4, units = 'in', res = 300)
    pheatmap::pheatmap(scm, cluster_rows = rowOrder, cluster_cols = colOrder,
                       breaks=c(seq(0,1,.02)), main = plotTitle, color = heatmapColors, 
                       border_color = NA, treeheight_row = 0, treeheight_col = 0,
                       show_rownames = FALSE)
    grDevices::dev.off()
  } else {
    pheatmap::pheatmap(scm, cluster_rows = rowOrder, cluster_cols = colOrder,
                       breaks=c(seq(0,1,.02)), main = plotTitle, color = heatmapColors, 
                       border_color = NA, treeheight_row = 0, treeheight_col = 0,
                       show_rownames = FALSE)
  }
}
