% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Intlr.R
\name{Intlr}
\alias{Intlr}
\title{Downscaling using interpolation and linear regression.}
\usage{
Intlr(
  exp,
  obs,
  exp_cor = NULL,
  exp_lats,
  exp_lons,
  obs_lats,
  obs_lons,
  lr_method,
  target_grid = NULL,
  points = NULL,
  int_method = NULL,
  method_point_interp = NULL,
  source_file_exp = NULL,
  source_file_obs = NULL,
  predictors = NULL,
  lat_dim = "lat",
  lon_dim = "lon",
  sdate_dim = "sdate",
  time_dim = "time",
  member_dim = "member",
  region = NULL,
  large_scale_predictor_dimname = "vars",
  loocv = TRUE,
  ncores = NULL
)
}
\arguments{
\item{exp}{an array with named dimensions containing the experimental field on the
coarse scale for which the downscaling is aimed. The object must have, at least,
the dimensions latitude, longitude and start date. The object is expected to be
already subset for the desired region. Data can be in one or two integrated regions, e.g.,
crossing the Greenwich meridian. To get the correct results in the latter case,
the borders of the region should be specified in the parameter 'region'. See parameter
'region'.}

\item{obs}{an array with named dimensions containing the observational field. The object
must have, at least, the dimensions latitude, longitude and start date. The object is
expected to be already subset for the desired region.}

\item{exp_cor}{an optional 's2dv_cube' object with named dimensions containing the seasonal
forecast experiment data. If provided, the forecast will be downscaled using the hindcast
and observations; if not, the hindcast will be downscaled instead. The default value is NULL.
Since the Intlr function is built separately for each ensemble member, it is not recommended
for forecast cases where the member_dim length of exp_cor differs from that of exp.
In such situations, the use of other functions in the package is more appropriate.}

\item{exp_lats}{a numeric vector containing the latitude values in 'exp'. Latitudes must
range from -90 to 90.}

\item{exp_lons}{a numeric vector containing the longitude values in 'exp'. Longitudes
can range from -180 to 180 or from 0 to 360.}

\item{obs_lats}{a numeric vector containing the latitude values in 'obs'. Latitudes must
range from -90 to 90.}

\item{obs_lons}{a numeric vector containing the longitude values in 'obs'. Longitudes
can range from -180 to 180 or from 0 to 360.}

\item{lr_method}{a character vector indicating the linear regression method to be applied.
Accepted methods are 'basic', 'large-scale' and '9nn'. The 'basic' method fits a
linear regression using high resolution observations as predictands and the
interpolated model data as predictor. Then, the regression equation is applied to the
interpolated model data to correct the interpolated values. The 'large-scale' method
fits a linear regression with large-scale predictors (e.g. teleconnection indices) as
predictors and high-resolution observations as predictands.
Finally, the '9nn' method uses a linear regression
with the nine nearest neighbours as predictors and high-resolution observations as
predictands. Instead of constructing a regression model using all nine predictors,
principal component analysis is applied to the data of neighboring grids to reduce the
dimension of the predictors. The linear regression model is then built using the principal
components that explain 95\% of the variance. The '9nn' method does not require a
pre-interpolation process.}

\item{target_grid}{a character vector indicating the target grid to be passed to CDO.
It must be a grid recognised by CDO or a NetCDF file.}

\item{points}{a list of two elements containing the point latitudes and longitudes
of the locations to downscale the model data. The list must contain the two elements
named as indicated in the parameters 'lat_dim' and 'lon_dim'. If the downscaling is
to a point location, only regular grids are allowed for exp and obs. Only needed if the
downscaling is to a point location.}

\item{int_method}{a character vector indicating the regridding method to be passed
to CDORemap. Accepted methods are "con", "bil", "bic", "nn", "con2". If "nn" method is
to be used, CDO_1.9.8 or newer version is required. For method "con2",
CDO_2.2.2 or older version is required.}

\item{method_point_interp}{a character vector indicating the interpolation method to
interpolate model gridded data into the point locations. Accepted methods are "nearest",
"bilinear", "9point", "invdist4nn", "NE", "NW", "SE", "SW".}

\item{source_file_exp}{a character vector with a path to an example file of the exp data.
Only needed if the downscaling is to a point location.}

\item{source_file_obs}{a character vector with a path to an example file of the obs data.
Only needed if the downscaling is to a point location.}

\item{predictors}{an array with large-scale data to be used in the 'large-scale' method.
Only needed if the linear regression method is set to 'large-scale'. It must have, at
least the dimension start date and another dimension whose name has to be specified in
the parameter 'large_scale_predictor_dimname'. It should contain as many elements as the
number of large-scale predictors.}

\item{lat_dim}{a character vector indicating the latitude dimension name in the element 'data'
in exp and obs. Default set to "lat".}

\item{lon_dim}{a character vector indicating the longitude dimension name in the element 'data'
in exp and obs. Default set to "lon".}

\item{sdate_dim}{a character vector indicating the start date dimension name in the element
'data' in exp and obs. Default set to "sdate".}

\item{time_dim}{a character vector indicating the time dimension name in the element
'data' in exp and obs. Default set to "time".}

\item{member_dim}{a character vector indicating the member dimension name in the element
'data' in exp and obs. Default set to "member".}

\item{region}{a numeric vector indicating the borders of the downscaling region.
It consists of four elements in this order: lonmin, lonmax, latmin, latmax. lonmin refers
to the left border, while lonmax refers to the right border. latmin indicates the lower
border, whereas latmax indicates the upper border. If set to NULL (default), the function
takes the first and last elements of the latitudes and longitudes in obs.}

\item{large_scale_predictor_dimname}{a character vector indicating the name of the
dimension in 'predictors' that contain the predictor variables. See parameter 'predictors'.}

\item{loocv}{a logical indicating whether to apply leave-one-out cross-validation when
generating the linear regressions. In this procedure, all values from the corresponding
year are excluded, so that when building the regression model for a given year, none of
that year’s data are used. Default to TRUE.}

\item{ncores}{an integer indicating the number of cores to use in parallel computation.
The default value is NULL.}
}
\value{
A list of three elements. 'data' contains the dowscaled field, 'lat' the
downscaled latitudes, and 'lon' the downscaled longitudes.
}
\description{
This function performs a downscaling using an interpolation and a linear
regression. Different methodologies that employ linear regressions are available. See
parameter 'lr_method' for more information. It is recommended that the observations
are passed already in the target grid. Otherwise, the function will also perform an
interpolation of the observed field into the target grid. The coarse scale and
observation data can be either global or regional. In the latter case, the region is
defined by the user. In principle, the coarse and observation data are intended to
be of the same variable, although different variables can also be admitted.
}
\examples{
\donttest{
exp <- rnorm(500) 
dim(exp) <- c(member = 5, lat = 4, lon = 5, sdate = 5) 
exp_lons <- 1:5 
exp_lats <- 1:4 
obs <- rnorm(900) 
dim(obs) <- c(lat = 12, lon = 15, sdate = 5) 
obs_lons <- seq(1,5, 4/14) 
obs_lats <- seq(1,4, 3/11)
if (Sys.which("cdo") != "") { 
res <- Intlr(exp = exp, obs = obs, exp_lats = exp_lats, exp_lons = exp_lons, 
            obs_lats = obs_lats, obs_lons = obs_lons, target_grid = 'r1280x640', 
            lr_method = 'basic', int_method = 'conservative')
}
}
}
\author{
J. Ramon, \email{jaumeramong@gmail.com}

E. Duzenli, \email{eren.duzenli@bsc.es}
}
