

<!-- README.md is generated from README.qmd. Please edit that file -->

# CatastRo <a href="https://ropenspain.github.io/CatastRo/"><img src="man/figures/logo.png" alt="CatastRo website" align="right" height="139"/></a>

<!-- badges: start -->

[![rOS-badge](https://ropenspain.github.io/rostemplate/reference/figures/ropenspain-badge.svg)](https://ropenspain.es/)
[![CRAN-status](https://www.r-pkg.org/badges/version/CatastRo)](https://CRAN.R-project.org/package=CatastRo)
[![CRAN-results](https://badges.cranchecks.info/worst/CatastRo.svg)](https://cran.r-project.org/web/checks/check_results_CatastRo.html)
[![Downloads](https://cranlogs.r-pkg.org/badges/CatastRo)](https://CRAN.R-project.org/package=CatastRo)
[![r-universe](https://ropenspain.r-universe.dev/badges/CatastRo)](https://ropenspain.r-universe.dev/CatastRo)
[![R-CMD-check](https://github.com/rOpenSpain/CatastRo/actions/workflows/roscron-check-standard.yaml/badge.svg)](https://github.com/rOpenSpain/CatastRo/actions/workflows/roscron-check-standard.yaml)
[![codecov](https://codecov.io/gh/rOpenSpain/CatastRo/graph/badge.svg?token=KPPwTkZjW6)](https://app.codecov.io/gh/rOpenSpain/CatastRo)
[![DOI](https://img.shields.io/badge/DOI-10.32614/CRAN.package.CatastRo-blue)](https://doi.org/10.32614/CRAN.package.CatastRo)
[![Project Status: Active – The project has reached a stable, usable
state and is being actively
developed.](https://www.repostatus.org/badges/latest/active.svg)](https://www.repostatus.org/#active)

<!-- badges: end -->

**CatastRo** is a package that provides access to different API services
of the [Spanish Cadastre](https://www.sedecatastro.gob.es/). With
**CatastRo**, you can download spatial objects such as buildings,
cadastral parcels, maps, and geocode cadastral references.

## Installation

<div class="pkgdown-release">

Install **CatastRo** from
[**CRAN**](https://CRAN.R-project.org/package=CatastRo):

``` r
install.packages("CatastRo")
```

</div>

<div class="pkgdown-devel">

Check the docs of the developing version in
<https://ropenspain.github.io/CatastRo/dev/>.

You can install the developing version of **CatastRo** using the
[r-universe](https://ropenspain.r-universe.dev/CatastRo):

``` r
# Install CatastRo in R:
install.packages(
  "CatastRo",
  repos = c(
    "https://ropenspain.r-universe.dev",
    "https://cloud.r-project.org"
  )
)
```

Alternatively, you can install the development version of **CatastRo**
with:

``` r
pak::pak("rOpenSpain/CatastRo")
```

</div>

## SSL issues

The SSL certificate of the Spanish Cadastre presents some issues that
may cause an error when using **CatastRo** (especially on macOS, see
issue [\#40](https://github.com/rOpenSpain/CatastRo/issues/40)):

In **CatastRo \>= 1.0.0** you can try to fix it by running this line in
your session right after you start using the package: \`

``` r
# Disable SSL verification
options(catastro_ssl_verify = 0)
```

If you wish to make this setup persistent, write the same code in your
[`.Rprofile`](https://docs.posit.co/ide/user/ide/guide/environments/r/managing-r.html):

``` r
# Open your .Rprofile with
usethis::edit_r_profile()

# And write on that file:
options(catastro_ssl_verify = 0)
```

## Package API

The functions of **CatastRo** are organised by API endpoint. The package
naming convention is `catr_*api*_*description*`.

### OVCCoordenadas

These functions allow geocoding and reverse geocoding of cadastral
references using the
[OVCCoordenadas](https://ovc.catastro.meh.es/ovcservweb/OVCSWLocalizacionRC/OVCCoordenadas.asmx)
service.

These functions are named `catr_ovc_get_*` and return a tibble, as
provided by the package **tibble**. See
`vignette("ovcservice", package = "CatastRo")` where these functions are
described.

### INSPIRE

These functions return spatial objects in the formats provided by the
**sf** or **terra** packages using the [Catastro
INSPIRE](https://www.catastro.hacienda.gob.es/webinspire/index.html)
service.

Note that the coverage of this service is 95% of the Spanish territory,
excluding the Basque Country and Navarre[^1], which have their own
independent cadastral offices.

There are three types of functions, each one querying a different
service:

#### ATOM service

The ATOM service allows batch-downloading vector objects of different
cadastral elements for a specific municipality. The result is provided
as `sf` objects (See **sf** package).

These functions are named `catr_atom_get_xxx`.

#### WFS service

The WFS service allows downloading vector objects of specific cadastral
elements. The results are provided as `sf` class objects (see the
[**sf**](https://r-spatial.github.io/sf/) package). Note that there are
some limitations on the extent and number of elements to query. For
batch downloading the ATOM service is preferred.

These functions are named `catr_wfs_get_xxx`.

#### WMS service

This service allows downloading georeferenced images of different
cadastral elements. The results are provided as rasters in the format
provided by the
[**terra**](https://rspatial.github.io/terra/reference/terra-package.html)
package.

There is a single function for querying this service:
`catr_wms_get_layer()`.

#### Terms and conditions of use

Please check the service’s [downloading
provisions](https://www.catastro.hacienda.gob.es/webinspire/documentos/Licencia.pdf).

## Examples

This script highlights some features of **CatastRo**:

### Geocode a cadastral reference

``` r
library(CatastRo)

catr_ovc_get_cpmrc(rc = "13077A01800039")
#> # A tibble: 1 × 10
#>   xcoord ycoord refcat     address pc.pc1 pc.pc2 geo.xcen geo.ycen geo.srs ldt  
#>    <dbl>  <dbl> <chr>      <chr>   <chr>  <chr>  <chr>    <chr>    <chr>   <chr>
#> 1  -3.46   38.6 13077A018… DS DIS… 13077… 18000… -3.4575… 38.6184… EPSG:4… DS D…
```

### Extract a cadastral reference from a given set of coordinates

``` r
catr_ovc_get_rccoor(
  lat = 38.6196566583596,
  lon = -3.45624183836806,
  srs = "4230"
)
#> # A tibble: 1 × 8
#>   refcat         address           pc.pc1 pc.pc2 geo.xcen geo.ycen geo.srs ldt  
#>   <chr>          <chr>             <chr>  <chr>     <dbl>    <dbl> <chr>   <chr>
#> 1 13077A01800039 DS DISEMINADO  P… 13077… 18000…    -3.46     38.6 EPSG:4… DS D…
```

### Extract geometries using the ATOM service

``` r
bu <- catr_atom_get_buildings("Nava de la Asuncion", to = "Segovia")

# Map
library(ggplot2)

ggplot(bu) +
  geom_sf(aes(fill = currentUse), col = NA) +
  coord_sf(
    xlim = c(374500, 375500),
    ylim = c(4556500, 4557500)
  ) +
  scale_fill_manual(values = hcl.colors(6, "Dark 3")) +
  theme_minimal() +
  labs(title = "Nava de la Asunción, Segovia")
```

<img src="man/figures/README-atom-1.png" style="width:100.0%"
alt="Extracting buildings in Nava de la Asuncion with the ATOM service" />

### Extract geometries using the WFS service

``` r
wfs_get_buildings <- catr_wfs_get_buildings_bbox(
  c(-4.134, 40.952, -4.131, 40.953),
  srs = 4326
)

# Map
ggplot(wfs_get_buildings) +
  geom_sf() +
  labs(title = "Alcázar of Segovia, Segovia, Spain")
```

<img src="man/figures/README-wfs-1.png" style="width:100.0%"
alt="Extract Alcázar of Segovia with the WFS service" />

## A note on caching

Some datasets and tiles may have a size larger than 50MB. You can use
**CatastRo** to create your own local repository at a given local
directory passing the following option:

``` r
catr_set_cache_dir("./path/to/location")
```

When this option is set, **CatastRo** will look for the cached file and
load it, speeding up the process.

## Citation

<p>

Delgado Panadero Á, Hernangómez D (2026). <em>CatastRo: Interface to the
API Sede Electrónica Del Catastro</em>.
<a href="https://doi.org/10.32614/CRAN.package.CatastRo">doi:10.32614/CRAN.package.CatastRo</a>,
<a href="https://ropenspain.github.io/CatastRo/">https://ropenspain.github.io/CatastRo/</a>.
</p>

A BibTeX entry for LaTeX users is:

    @Manual{R-CatastRo,
      title = {{CatastRo}: Interface to the {API} Sede Electrónica Del Catastro},
      author = {Ángel {Delgado Panadero} and Diego Hernangómez},
      doi = {10.32614/CRAN.package.CatastRo},
      year = {2026},
      version = {1.0.0},
      url = {https://ropenspain.github.io/CatastRo/},
      abstract = {Access public spatial data available under the INSPIRE directive. Tools for downloading references and addresses of properties, as well as map images.},
    }

## Contribute

Check the GitHub page for the [source
code](https://github.com/ropenspain/CatastRo/).

[^1]: The package
    [**CatastRoNav**](https://ropenspain.github.io/CatastRoNav/)
    provides access to the Cadastre of Navarre, with similar
    functionalities to **CatastRo**.
