#' Incremental Perturbation PCA
#'
#' Batch-updated PCA with residual-projection perturbation.
#'
#' @param batch_data List of matrices.
#' @param m Components.
#' @param eta Step size.
#' @return List with Ap, Dp, V.
#' @export
#' @examples
#' \dontrun{
#' set.seed(789)
#' N <- 50; m.true <- 2; n_batch <- 5
#' batches <- lapply(1:n_batch, function(i) matrix(rnorm(60 * N), 60, N))
#' inc <- EFM::incremental_perturb_pca(batches, m = m.true)
#' print(round(inc$Ap[1:5, 1:2], 3))
#' }
incremental_perturb_pca <- function(batch_data, m, eta = 0.1) {
  p <- ncol(batch_data[[1]])
  X1 <- scale(batch_data[[1]])
  S1 <- cov(X1)
  V <- eigen(S1)$vectors[, order(eigen(S1)$values, decreasing = TRUE)[1:m]]
  for (t in 2:length(batch_data)) {
    Xt <- scale(batch_data[[t]])
    nt <- nrow(Xt)
    P <- diag(p) - V %*% t(V)
    DV <- (t(Xt) %*% P %*% Xt) / nt
    dv <- svd(DV)$u[, 1, drop = FALSE]
    V <- V + eta * dv
    V <- qr.Q(qr(V))
  }
  Xall <- do.call(rbind, batch_data)
  Xall <- scale(Xall)
  Sall <- cov(Xall)
  lam <- diag(t(V) %*% Sall %*% V)
  Ap <- V %*% diag(sqrt(lam))
  h2 <- diag(Ap %*% t(Ap))
  Dp <- diag(Sall) - h2
  list(Ap = Ap, Dp = Dp, V = V)
}