% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trend.R
\name{make_trend}
\alias{make_trend}
\title{Create a trend specification for model parameters}
\usage{
make_trend(
  par_names,
  cov_names = NULL,
  kernels,
  bases = NULL,
  shared = NULL,
  trend_pnames = NULL,
  phase = "premap",
  par_input = NULL,
  at = "lR",
  maps = NULL,
  custom_trend = NULL,
  ffill_na = NULL
)
}
\arguments{
\item{par_names}{Character vector specifying which parameters to apply trend to}

\item{cov_names}{Character vector specifying which covariates to use for each trend}

\item{kernels}{Character vector specifying which kernel function to use for each trend}

\item{bases}{Optional character vector specifying which base function to use for each trend}

\item{shared}{Named list with entries the parameter names to be shared and the names the new names of the shared parameter.}

\item{trend_pnames}{Optional character vector specifying custom parameter names}

\item{phase}{Character vector (length 1 or \code{length(par_names)}) specifying the phase for each trend entry;
one of "premap", "pretransform", or "posttransform". Defaults to "premap".}

\item{par_input}{Optional character vector(s) of parameter names to use as additional inputs for the trend}

\item{at}{If NULL, trend is applied to every row in the \code{dadm}. If a factor name (e.g., "lR"), trend is applied only to entries
corresponding to the first level of that factor, and fed forward to the other levels of that factor. Defaults to "lR". For DDMs, \code{at} should be set to NULL.}

\item{maps}{List of functions that create matrices with which to multiply the covariates before applying the base. See details.}

\item{custom_trend}{A trend registered with \code{register_trend}}

\item{ffill_na}{Determines how missing covariate values are handled.
If \code{TRUE}, missing values are forward-filled using the last known non-\code{NA} value after
applying the kernel. If \code{FALSE}, trials with missing covariates contribute \code{0} instead.
The default (NULL) is interpreted as \code{TRUE} for delta-rule models and \code{FALSE} otherwise.}
}
\value{
A list containing the trend specifications for each parameter
}
\description{
Create a trend specification for model parameters
}
\details{
The \code{maps} argument accepts one or more functions that translate trial-level
covariates into accumulator-specific predictors.

Example of a minimal map function:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{advantage_map <- function(dadm, cov_names) \{
  lS      <- paste0('cov', ifelse(dadm$lR == 'left',  dadm$cov_left,  dadm$cov_right))
  lSother <- paste0('cov', ifelse(dadm$lR == 'right', dadm$cov_left,  dadm$cov_right))
  plus  <- sapply(cov_names, function(x) ifelse(lS      == x,  1, 0))
  minus <- sapply(cov_names, function(x) ifelse(lSother == x, -1, 0))
  plus + minus
\}
}\if{html}{\out{</div>}}

Multiple maps may be supplied, in which case the model will create a separate
base parameter for each map. See more examples below.
}
\examples{
# Put trend on B and v parameters
trend <- make_trend(
  par_names = c("B", "v"),
  cov_names = "strial",
  kernels = c("exp_incr", "poly3"),
  phase = "premap",
  shared = list(shrd = list("B.B0", "v.d1"))
)
get_trend_pnames(trend)


# Using covariate maps

# Covariate maps allow you to specify how trial-by-trial covariates influence
# model parameters for each accumulator. The example below uses a simple data
# frame with two trials. `cov_left` and `cov_right` specify which covariates
# correspond to the left and right accumulators on each trial. `S` indicates the
# correct response, and `cov1`–`cov4` contain the actual covariate values.

data <- data.frame(
  subjects = rep(1, 2),
  S        = c('left', 'right'),
  cov_left = c('1', '4'),
  cov_right= c('3', '2'),
  rt       = c(1.2, 0.8),
  R        = factor(c('left', 'right')),
  cov1     = c(1, NA),
  cov2     = c(NA, 1),
  cov3     = c(NA, 1),
  cov4     = c(1, 1)
)

# A covariate map function must take `dadm` and `cov_names` as inputs and return
# a matrix of size (nrow(dadm), length(cov_names)), coding how each covariate
# contributes to each accumulator.

advantage_map <- function(dadm, cov_names) {

  # Which stimulus does the accumulator correspond to on each trial?
  lS <- paste0('cov', ifelse(dadm$lR == 'left', dadm$cov_left, dadm$cov_right))

  # Which stimulus does the *other* accumulator correspond to?
  lSother <- paste0('cov', ifelse(dadm$lR == 'right', dadm$cov_left, dadm$cov_right))

  # Build indicator matrices
  map_plus1 <- sapply(cov_names, function(col) ifelse(lS     == col,  1, 0))
  map_minus1<- sapply(cov_names, function(col) ifelse(lSother == col, -1, 0))

  map_plus1 + map_minus1
}

# A covariate map function can be supplied to make_trend(), which creates the mapping
# specification for the model for each participant. Here, a single map ('differences') is provided.

trend <- make_trend(
  par_names = 'v',
  kernels   = 'delta',
  bases     = 'lin',
  cov_names = list(c('cov1', 'cov2', 'cov3', 'cov4')),
  maps      = list('differences' = advantage_map),
  at        = 'lR'
)

design_RDM <- design(
  model  = RDM,
  data   = data,
  formula= list(B ~ 1, v ~ 1, t0 ~ 1),
  trend  = trend
)

emc <- make_emc(data, design_RDM, type = 'single')

# The resulting covariate maps for each subject are attached to the `dadm`:
attr(emc[[1]]$data[[1]], 'covariate_maps')
# And to confirm that this mapping is correct, compare with the corresponding `dadm`
emc[[1]]$data[[1]]

# You can also provide multiple covariate maps. Each additional map introduces
# a separate base parameter. For example, the following `sum_map` is suitable
# for RL-ARD–type models:

sum_map <- function(dadm, cov_names) {
  # Which stimulus does the accumulator correspond to on each trial?
  lS <- paste0('cov', ifelse(dadm$lR == 'left', dadm$cov_left, dadm$cov_right))
  # Which stimulus does the *other* accumulator correspond to?
  lSother <- paste0('cov', ifelse(dadm$lR == 'right', dadm$cov_left, dadm$cov_right))

  # Indicator matrices (note: both are added rather than subtracted)
  map_this  <- sapply(cov_names, function(col) ifelse(lS      == col, 1, 0))
  map_other <- sapply(cov_names, function(col) ifelse(lSother == col, 1, 0))

  map_this + map_other
}

trend <- make_trend(
  par_names = 'v',
  kernels   = 'delta',
  bases     = 'lin',
  cov_names = list(c('cov1', 'cov2', 'cov3', 'cov4')),
  maps      = list('differences' = advantage_map,
                   'sums'        = sum_map),
  at        = 'lR'
)

design_RDM <- design(
  model  = RDM,
  data   = data,
  formula= list(B ~ 1, v ~ 1, t0 ~ 1),
  trend  = trend
)

emc <- make_emc(data, design_RDM, type = 'single')

# Now the dadm contains two covariate maps, and the model includes two
# corresponding base parameters (e.g., v.w1 and v.w2):
attr(emc[[1]]$data[[1]], 'covariate_maps')


}
