% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wrAdd.R
\name{wrAdd}
\alias{wrAdd}
\alias{wrAdd.default}
\alias{wrAdd.formula}
\title{Computes a vector of relative weights specific to a species in an entire data frame.}
\usage{
wrAdd(wt, ...)

\method{wrAdd}{default}(
  wt,
  len,
  spec,
  thesaurus = NULL,
  units = c("metric", "English"),
  WsOpts = NULL,
  ...
)

\method{wrAdd}{formula}(wt, data, thesaurus = NULL, units = c("metric", "English"), ...)
}
\arguments{
\item{wt}{A numeric vector that contains weight measurements or a formula of the form \code{wt~len+spec} where \dQuote{wt} generically represents the weight variable, \dQuote{len} generically represents the length variable, and \dQuote{spec} generically represents the species variable. Note that this formula can only contain three variables and they must be in the order of weight first, length second, species third.}

\item{\dots}{Not used.}

\item{len}{A numeric vector that contains length measurements. Not used if \code{wt} is a formula.}

\item{spec}{A character or factor vector that contains the species names. Not used if \code{wt} is a formula.}

\item{thesaurus}{A named list for providing alternative species names (the values in the list) that correspond to specific names in \code{PSDlit} (the names in the list). See details and examples.}

\item{units}{A string that indicates whether the weight and length data in \code{formula} are in \code{"metric"} (DEFAULT; mm and g) or \code{"English"} (in and lbs) units.}

\item{WsOpts}{A named list that provides specific choices for \code{group}, \code{ref}, or \code{method} for species for which more than one standard weight equation exists in \code{\link{WSlit}}.}

\item{data}{A data.frame that minimally contains variables of the the observed lengths, observed weights, and the species names given in the \code{formula=}.}
}
\value{
A numeric vector that contains the computed relative weights, in the same order as in \code{data=}.
}
\description{
Returns a vector that contains the relative weight specific to each species for all individuals in an entire data frame.
}
\details{
This computes a vector that contains the relative weight specific to each species for all individuals in an entire data frame. The vector can be appended to an existing data.frame to create a variable that contains the relative weights for each individual. The relative weight value will be \code{NA} for each individual for which a standard weight equation does not exist in \code{\link{WSlit}}, a standard weight equation for the units given in \code{units=} does not exist in \code{\link{WSlit}}, or if the individual is shorter or longer than the lengths for which the standard weight equation should be applied. Either the linear or quadratic equation has been listed as preferred for each species, so only that equation will be used.

The species names in \code{species} must match the spelling and capitalization of \code{species} in \code{\link{WSlit}}. Use \code{wsVal()} to see a list of all species for which standard weight equations exist in \code{\link{WSlit}} and, more importantly, how the species names are spelled and capitalized.

The \code{thesaurus} argument may be used to relate alternate species names to the species names used in \code{WSlit}. For example, you (or your data) may use \dQuote{Bluegill Sunfish}, but \dQuote{Bluegill} is used in \code{WSlit}. The alternate species name can be used here if it is defined in a named vector (or list) given to \code{thesarus=}. The alternate species name is the value and the species name in \code{PSDlit} is the name in this vector/list - e.g., \code{c("Bluegill"="Bluegill Sunfish")}. See the examples for a demonstration.

Some species have length categories separated by sub-group. For example, length categories exist for both lentic and lotic populations of Brown Trout. The length values for a sub-group may be obtained by either including the species name in \code{species} and the sub-group name in \code{group} in \code{WsOpts} or by using the combined species and sub-group name, with the sub-group name in parentheses, in \code{species}. Both methods are demonstrated in the examples. Note that an error is returned if a species has sub-groups but neither method is used to define the sub-group.

Some (few) species have more than one equation listed in \code{\link{WSlit}} (for the specified units). In these instances the user must select one of the equations to use with \code{WsOpts}. \code{WsOpts} is a list of lists where the inside list contains one or more of \code{group}, \code{ref}, or \code{method} (see \code{\link{WSlit}}) required to specify a single equation for a particular species, which is the name of the inner list. See the examples for an illustration of how to use \code{WsOpts}.

See examples and \href{https://fishr-core-team.github.io/FSA/articles/Computing_Relative_Weights.html}{this article} for a demonstration.
}
\section{IFAR Chapter}{
 8-Condition.
}

\examples{
#===== Simple example with 3 species, 2 in WSlit ... nothing unusual
tmp <- subset(PSDWRtest,
              species \%in\% c("Yellow Perch","Iowa Darter","Largemouth Bass"),
              select=c("species","len","wt"))
peek(tmp,n=10)

#----- Add Wr variable ... using formula interface
tmp$wr1 <- wrAdd(wt~len+species,data=tmp)
#----- same but with non-formula interface
tmp$wr2 <- wrAdd(tmp$wt,tmp$len,tmp$species)
#----- same but using dplyr
if (require(dplyr)) {
  tmp <- tmp \%>\%
    mutate(wr3=wrAdd(wt,len,species))
}
#----- examine results
peek(tmp,n=10)

#===== Simple example with only one species in the data.frame
tmp <- subset(PSDWRtest,species \%in\% c("Yellow Perch"),
              select=c("species","len","wt"))
tmp$wr <- wrAdd(wt~len+species,data=tmp)
peek(tmp,n=6)

#===== Example of species with sub-groups but only 1 sub-group in data.frame
#-----   Group not in species name so must specify group with WsOpts
tmp <- subset(PSDWRtest,species=="Brown Trout" & location=="Trout Lake",
              select=c("species","len","wt"))
tmp$wr1 <- wrAdd(wt~len+species,data=tmp,
                 WsOpts=list("Brown Trout"=list("group"="lotic")))

#-----   Group in species name so don't specify group with WsOpts
tmp$species2 <- "Brown Trout (lotic)"
tmp$wr2 <- wrAdd(wt~len+species2,data=tmp)  # note use of species2

peek(tmp,n=6)

#===== Example of species with sub-groups and 2 sub-groups in data.frame
tmp <- subset(PSDWRtest,species=="Brown Trout",
              select=c("species","location","len","wt"))
#-----   Must create "species" with sub-groups in name
#-----     Many ways to do this, this is just one example for this case
tmp$species2 <- ifelse(tmp$location=="Trout Lake",
                       "Brown Trout (lotic)","Brown Trout (lentic)")
tmp$wr <- wrAdd(wt~len+species2,data=tmp)  # note use of species2
peek(tmp,n=6)

#===== Example of a species name that needs the thesaurus
tmp <- subset(PSDWRtest,species \%in\% c("Yellow Perch","Bluegill Sunfish"),
              select=c("species","len","wt"))
#-----  Below will not add wr for "Bluegill Sunfish" as not in WsLit ("Bluegill" is)
tmp$wr1 <- wrAdd(wt~len+species,data=tmp)
#-----  Use thesaurus to identify "Bluegill Sunfish" as "Blueill
tmp$wr2 <- wrAdd(wt~len+species,data=tmp,thesaurus=c("Bluegill"="Bluegill Sunfish"))
peek(tmp,n=10)

#===== Example of species that has Ws eqns for multiple reference values
tmp <- subset(PSDWRtest,species=="Ruffe",select=c("species","len","wt"))
#-----  Below will err as Ruffe has Ws eqns for multiple reference values
# tmp$wr <- wrAdd(wt~len+species,data=tmp)
#-----  Must choose which eqn to use with WsOpts
tmp$wr <- wrAdd(wt~len+species,data=tmp,
                WsOpts=list(Ruffe=list(ref=75)))
peek(tmp,n=6)

#===== Example with two uses of WsOpts (and one species without)
tmp <- subset(PSDWRtest,species \%in\% c("Ruffe","Muskellunge","Iowa Darter"),
              select=c("species","len","wt"))
tmp$wr <- wrAdd(wt~len+species,data=tmp,
                WsOpts=list(Muskellunge=list(group="overall"),
                            Ruffe=list(ref=75)))
peek(tmp,n=10)

}
\references{
Ogle, D.H. 2016. \href{https://fishr-core-team.github.io/fishR/pages/books.html#introductory-fisheries-analyses-with-r}{Introductory Fisheries Analyses with R}. Chapman & Hall/CRC, Boca Raton, FL.
}
\seealso{
See \code{\link{wsVal}}, \code{\link{WSlit}}, and \code{\link{psdAdd}} for related functionality. See \code{\link[plyr]{mapvalues}} for help in changing species names to match those in \code{\link{WSlit}}.
}
\author{
Derek H. Ogle, \email{DerekOgle51@gmail.com}
}
\keyword{manip}
