# R/caps.R

#' Internal helper: volatility-aware asymmetric caps
#'
#' Applies volatility-aware asymmetric caps to raw growth signals.
#' Key stabilizations:
#'   • prevents collapse to zero-growth band,
#'   • ensures minimum cap width,
#'   • restricts extreme caps (never > ±50%),
#'   • frequency-aware volatility scaling.
#'
#' @param growth_raw Numeric vector of raw growth signals.
#' @param base_low Baseline lower cap (default -0.3).
#' @param base_high Baseline upper cap (default +0.3).
#' @param freq_name Optional frequency label ("week","month","quarter","year").
#' @param freq_value Optional ts frequency integer.
#'
#' @return Numeric vector of capped growth signals.
#'
#' @keywords internal
#' @noRd
#' @importFrom stats sd
.gace_apply_volatility_caps <- function(growth_raw,
                                        base_low   = -0.3,
                                        base_high  =  0.3,
                                        freq_name  = NULL,
                                        freq_value = NULL) {
  
  # ---------------------------------------------------------
  # Safety: empty input
  # ---------------------------------------------------------
  if (length(growth_raw) == 0L) return(growth_raw)
  
  # Normalize freq_name safely
  freq_name <- tolower(ifelse(is.null(freq_name), "", freq_name))
  
  # ---------------------------------------------------------
  # Frequency-aware scaling factor (k)
  # ---------------------------------------------------------
  k <- 1.2  # default
  
  if (freq_name %in% c("week", "weekly", "w"))      k <- 1.0
  if (freq_name %in% c("quarter", "qtr", "q"))      k <- 1.4
  if (freq_name %in% c("year", "yearly", "annual")) k <- 1.6
  
  # TS frequency overrides name-based choice
  if (!is.null(freq_value)) {
    if (freq_value == 12L) k <- 1.2
    if (freq_value >= 50L) k <- 1.0
    if (freq_value == 4L)  k <- 1.4
  }
  
  # ---------------------------------------------------------
  # Volatility estimation
  # ---------------------------------------------------------
  vol <- stats::sd(growth_raw, na.rm = TRUE)
  if (!is.finite(vol)) vol <- 0
  
  # ---------------------------------------------------------
  # Step 1 — compute initial volatility-based caps
  # ---------------------------------------------------------
  low  <- max(base_low,  -k * vol)
  high <- min(base_high,  k * vol)
  
  # ---------------------------------------------------------
  # Step 2 — enforce a minimum width (prevents zero-band collapse)
  # ---------------------------------------------------------
  min_width <- 0.05
  
  if ((high - low) < min_width) {
    center <- (high + low) / 2
    low  <- center - min_width / 2
    high <- center + min_width / 2
  }
  
  # ---------------------------------------------------------
  # Step 3 — global hard limits (±50%)
  # ---------------------------------------------------------
  hard_low  <- -0.50
  hard_high <-  0.50
  
  low  <- max(low,  hard_low)
  high <- min(high, hard_high)
  
  # ---------------------------------------------------------
  # Step 4 — apply caps
  # ---------------------------------------------------------
  pmin(pmax(growth_raw, low), high)
}