#' Function to perform an EM algorithm (logistic prior \eqn{P(Z = 1 \mid v)} + beta p-values (discriminative prior))
#' for a 2-component mixture where each observation \eqn{i} has a p-value \eqn{p_i}, and an auxiliary feature vector \eqn{v_i}.
#' @param pvalues Vector of gene-level p-values.
#' @param v An auxiliary feature vector.
#' @param alpha0 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param alpha1 Shape parameter of the alternative/signal p-value distribution, modeled as
#'   \eqn{p \sim \mathrm{Beta}(\alpha_1, 1)} when \eqn{Z = 1}; values \eqn{< 1} concentrate mass near 0 (enrichment of small p-values).
#' @param pi0 Mixing proportion (prior probability) that an observation belongs to the null/background component \eqn{Z = 0}
#'   in the two-component mixture.
#' @param pi1 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param max.it Maximum number of iterations for the EM algorithm.
#' @param verbose A logical value defining if the iteration information should be printed or not.
#' @return Returns the posterior probability of association for each gene.
#' @name model.LR
#' @keywords internal
model.LR <- function(pvalues,v, lambda,alpha0, alpha1,pi0,pi1, max.it, verbose=verbose){
  
  n <- dim(v)[1]
  d <- dim(v)[2]

  r <- EMinfer(pvalues,pi0,pi1, alpha0, alpha1, max.it)
  w <- seq(0,0,length.out = d)
  alpha0 <- r$alpha0
  alpha1 <- r$alpha1
  b <- 0
  z <- r$post
  gamma <- 1
  
  loglik.history <- c()
  i <- 0
  while(1){
    i <- i + 1
    prior <- 1/(1+exp(- v %*% w - b))
    prior <- prior[,1]
    
    tmp0 <- log(1 - prior) + log(alpha0) + (alpha0 - 1) * log(pvalues)
    tmp1 <- log(prior) + log(alpha1) + (alpha1 - 1) * log(pvalues)
    z <- 1 / (1 + exp(tmp0 - tmp1))
    
    max.tmp <- max(c(tmp1,tmp0))
    loglik <- sum(log(exp(tmp1-max.tmp) + exp(tmp0-max.tmp)) + max.tmp) - lambda/2 * n* sum(w*w)
    loglik.history <- c(loglik.history,loglik)
    
    alpha1 <- - sum(z) / sum(z * log(pvalues))
    alpha0 <- - sum(1-z) / sum((1-z) * log(pvalues))
    
    grad.w <- apply(v * (z - gamma * prior + (gamma -1) * z * prior),2,sum) - lambda * n * w 
    grad.b <- sum(z - gamma * prior + (gamma - 1) * z * prior)
    
    H <- matrix(0,ncol = d+1,nrow = d+1)
    tmp <- prior*(prior-1) #* idx
    H[1:d,1:d] <- t(v) %*% (v * (gamma * tmp + (1-gamma) * z * tmp) ) - lambda * n * diag(d)
    H[d+1,d+1] <- sum(gamma * tmp + (1-gamma) * z * tmp)
    H[1:d,d+1] <- apply((v * (gamma * tmp + (1-gamma)*z*tmp ) ),2,sum)
    H[d+1,1:d] <- apply((v * (gamma * tmp + (1-gamma)*z*tmp ) ),2,sum)
    
    H <- H + 1e-6 * diag(d+1)
    step <- solve(H, c(grad.w,grad.b))
    w <- w - step[1:d] 
    b <- b - step[d+1]
    
    if(verbose) print(sprintf('iter %02d , log(likelihood) %.6f',i,loglik))
    
    if(i > 10 && abs(loglik.history[i-10] - loglik.history[i]) < 1e-6 ) break;
    if(i >= 500) break;
    if(i > 1 && loglik.history[i] < loglik.history[i-1]) break;
  }
  
  result <- list(loglik=loglik.history, alpha0=alpha0,alpha1=alpha1,w=w,b=b,post=z)
  
  return(result)
}
