#' Function to perform an EM algorithm (“Multiple” univariate KDEs (nonparametric naive Bayes)) for a 2-component mixture where each observation i has a p-value pi, and an auxiliary feature vector vi
#' @param pvalues Vector of gene-level p-values
#' @param v an auxiliary feature vector
#' @param alpha0 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param alpha1 Shape parameter of the alternative/signal p-value distribution, modeled as
#'   \eqn{p \sim \mathrm{Beta}(\alpha_1, 1)} when \eqn{Z = 1}; values \eqn{< 1} concentrate mass near 0 (enrichment of small p-values).
#' @param pi0 Mixing proportion (prior probability) that an observation belongs to the null/background component \eqn{Z = 0}
#'   in the two-component mixture.
#' @param pi1 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param max.it Maximum number of iterations for the EM algorithm.
#' @param verbose A logical value defining if the iteration information should be printed or not.
#' @return Returns the posterior probability of association for each gene.
#' @name model.M
#' @keywords internal
model.M <- function(pvalues,v,alpha0, alpha1,pi0,pi1, max.it, verbose=verbose){
  n <- dim(v)[1]
  d <- dim(v)[2]
  
  r <- EMinfer(pvalues,pi0,pi1, alpha0, alpha1, max.it)
  alpha0 <- r$alpha0
  alpha1 <- r$alpha1
  pi <- r$pi
  z <- r$post
  old.z <- z
  
  density0 <- matrix(0,nrow = n, ncol = d)
  density1 <- matrix(0,nrow = n, ncol = d)
  
  loglik.history <- c()
  i <- 0
  while(1){
    i <- i + 1
    
    for(k in 1:d){
      density0[,k] <- kde(v[,k],1-z)
      density1[,k] <- kde(v[,k],z)
    }
    
    # E-step
    tmp0 <- log(1-pi) + log(alpha0) + (alpha0-1)*log(pvalues) + apply(log(density0),1,sum)
    tmp1 <- log(pi) + log(alpha1) + (alpha1-1)*log(pvalues) + apply(log(density1),1,sum)
    z <- 1 / (1 + exp(tmp0 - tmp1))
    
    pi <- sum(z) / n
    alpha0 <- - sum(1-z) / sum((1-z) * log(pvalues))
    alpha1 <- - sum(z) / sum(z * log(pvalues))
    
    if(sum(abs(old.z - z)) < 1e-6) break;
    if(i > 1000) break;
    old.z <- z
  }
  
  result <- list(post=z, pi=pi, alpha0=alpha0, alpha1=alpha1)
  return(result)
}
