% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gareg_subset.R
\name{gareg_subset}
\alias{gareg_subset}
\title{Genetic-Algorithm Best Subset Selection (GA / GAISL)}
\usage{
gareg_subset(
  y,
  X,
  ObjFunc = NULL,
  gaMethod = "ga",
  gacontrol = NULL,
  monitoring = FALSE,
  seed = NULL,
  ...
)
}
\arguments{
\item{y}{Numeric response vector (length \code{n}).}

\item{X}{Numeric matrix of candidate predictors (\code{n} rows by \code{p} columns).}

\item{ObjFunc}{Objective function or its name. Defaults to \code{\link{subsetBIC}}.
The objective must accept as its first argument a binary chromosome
(0/1 mask of length \code{p}) and may accept additional arguments passed via \code{...}.
By convention, \code{subsetBIC} returns \emph{negative} BIC, so the GA maximizes fitness.}

\item{gaMethod}{GA backend to call: \code{"ga"} or \code{"gaisl"} (functions from package \pkg{GA}),
or a GA-compatible function with the same interface as \code{\link[GA]{ga}}.}

\item{gacontrol}{Optional named list of GA engine controls (e.g., \code{popSize}, \code{maxiter},
\code{run}, \code{pcrossover}, \code{pmutation}, \code{elitism}, \code{seed}, \code{parallel},
\code{keepBest}, \code{monitor}, ...). These are passed to the GA engine, not to the objective.}

\item{monitoring}{Logical; if \code{TRUE}, prints a short message and (if not supplied in
\code{gacontrol}) sets \code{monitor = GA::gaMonitor} for live progress.}

\item{seed}{Optional RNG seed (convenience alias for \code{gacontrol$seed}).}

\item{...}{Additional arguments forwarded to \code{ObjFunc} (not to the GA engine). For
\code{\link{subsetBIC}} these typically include \code{family}, \code{weights}, \code{offset},
and \code{control}.}
}
\value{
An object of S4 class \code{"gareg"} (with \code{method = "subset"}) containing:
\itemize{
  \item \code{call} – the matched call.
  \item \code{N} – number of observations.
  \item \code{objFunc} – the objective function used.
  \item \code{gaMethod} – \code{"ga"} or \code{"gaisl"}.
  \item \code{gaFit} – the GA fit object returned by \pkg{GA} (if your class allows it).
  \item \code{featureNames} – column names of \code{X} (or empty).
  \item \code{bestFitness} – best fitness value (\code{GA::ga}@fitnessValue).
  \item \code{bestChrom} – \code{c(m, idx)}: number of selected variables and their indices.
  \item \code{bestnumbsol} – \code{m}, number of selected variables.
  \item \code{bestsol} – vector of selected column indices in \code{X}.
}
}
\description{
Runs a GA-based search over variable subsets using a user-specified
objective (default: \code{\link{subsetBIC}}) and returns a compact
\code{"gareg"} S4 result with \code{method = "subset"}.
The engine can be \code{\link[GA]{ga}} (single population) or
\code{\link[GA]{gaisl}} (islands), selected via \code{gaMethod}.
}
\details{
The fitness passed to \pkg{GA} is \code{ObjFunc} itself. Because the engine expects
a function with signature \code{f(chrom, ...)}, your \code{ObjFunc} must interpret
\code{chrom} as a 0/1 mask over the columns of \code{X}. The function then computes a score
(e.g., negative BIC) using \code{y}, \code{X}, and any extra arguments supplied via \code{...}.

With the default \code{\link{subsetBIC}}, the returned value is \code{-BIC}, so we set
\code{max = TRUE} in the GA call to maximize fitness. If you switch to an objective that
returns a quantity to \emph{minimize}, either negate it in your objective or change
the engine setting to \code{max = FALSE}.

Engine controls belong in \code{gacontrol}; objective-specific options belong in \code{...}.
This separation prevents accidental name collisions between GA engine parameters and
objective arguments.
}
\examples{
\donttest{
if (requireNamespace("GA", quietly = TRUE)) {
  set.seed(1)
  n <- 100
  p <- 12
  X <- matrix(rnorm(n * p), n, p)
  y <- 1 + X[, 1] - 0.7 * X[, 4] + rnorm(n, sd = 0.5)

  # Default: subsetBIC (Gaussian – negative BIC), engine = GA::ga
  fit1 <- gareg_subset(y, X,
    gaMethod = "ga",
    gacontrol = list(popSize = 60, maxiter = 80, run = 40)
  )
  summary(fit1)

  # Island model: GA::gaisl
  fit2 <- gareg_subset(y, X,
    gaMethod = "gaisl",
    gacontrol = list(popSize = 40, maxiter = 60, islands = 4)
  )
  summary(fit2)

  # Logistic objective (subsetBIC handles GLM via ...):
  ybin <- rbinom(n, 1, plogis(0.3 + X[, 1] - 0.5 * X[, 2]))
  fit3 <- gareg_subset(ybin, X,
    gaMethod = "ga",
    family = stats::binomial(), # <- passed to subsetBIC via ...
    gacontrol = list(popSize = 60, maxiter = 80)
  )
  summary(fit3)
}
}

}
\seealso{
\code{\link{subsetBIC}},
\code{\link[GA]{ga}},
\code{\link[GA]{gaisl}}
}
