## ----setup, include = FALSE---------------------------------------------------
library(GCubeR)
library(dplyr)
library(knitr)
library(kableExtra)

knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = "center",
  fig.width = 7,
  fig.height = 5,
  message = FALSE,
  warning = FALSE
)
gembloux_kable <- function(x, caption = NULL, digits = 3) {
  # nombre de colonnes du data.frame (à garder avant le pipe)
  last_col <- ncol(x)

  pal_dark   <- "#004B87"  # dark blue
  pal_light  <- "#e8f5e9"  # light green
  pal_accent <- "#2e7d32"  # mid green

  x %>%
    kable(format = "html", digits = digits, caption = caption) %>%
    kable_styling(
      full_width        = FALSE,
      bootstrap_options = c("striped", "hover", "condensed")
    ) %>%
    row_spec(0, background = pal_dark, color = "white", bold = TRUE) %>%
    column_spec(1, background = pal_light, bold = TRUE) %>%
    column_spec(last_col, background = pal_accent, color = "white", bold = TRUE)
}

## ----echo=FALSE, out.width="100%", fig.align="center"-------------------------
knitr::include_graphics("img/function_diag.png")

## ----echo=TRUE, message=FALSE, warning=FALSE----------------------------------

# Data creation
data <- data.frame(
  species_code = c("LARIX_SP","LARIX_SP","LARIX_SP","LARIX_SP"),
  htot = c(12,13,10,11),
  c130 = c(65,56,52,68)
)

# Computes volumes (function from package)
data <- rondeux_vc22_vtot(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("QUERCUS_SP","QUERCUS_SP","QUERCUS_SP","QUERCUS_SP"),
  htot = c(28,35,21,36),
  c130 = c(180,235,175,238)
)
# Adding dbh
data <- add_c130_dbh(data)

# Computes volumes
data <- bouvard_vta(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","ABIES_ALBA","ABIES_ALBA","ABIES_ALBA"),
  htot = c(28,35,21,36),
  c130 = c(180,235,175,238)
)

# Adding dbh
data <- add_c130_dbh(data)

# Computes volumes
data <- algan_vta_vc22(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_PINASTER","ABIES_ALBA"),
  htot = c(28,35,21,36),
  c130 = c(180,235,175,238)
)

# Computes volumes
data <- vallet_vta(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_SYLVESTRIS","ABIES_ALBA"),
  htot = c(28,35,21,36),
  c130 = c(180,235,175,238)
)

# Computes volumes
data <- dagnelie_vc22_1(data)

## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_SYLVESTRIS","ABIES_ALBA"),
  hdom = c(20,25,18,35),
  c130 = c(180,235,175,238)
)

# Computes volumes
data <- dagnelie_vc22_1g(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_SYLVESTRIS","ABIES_ALBA"),
  htot = c(20,25,18,35),
  c130 = c(180,235,175,238)
)

# Computes volumes
data <- dagnelie_vc22_2(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_PINASTER","ABIES_ALBA"),
  htot = c(28,35,21,36),
  c130 = c(180,235,175,238)
)

# Adding dbh
data <- add_c130_dbh(data)

# Computes volumes
data <- vallet_vc22(data)

## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("LARIX_SP","LARIX_SP","LARIX_SP","LARIX_SP"),
  htot = c(12,13,10,11),
  c130 = c(65,56,52,68)
)

# Computes volumes
data <- rondeux_vc22_vtot(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","ABIES_ALBA","ABIES_ALBA","ABIES_ALBA"),
  htot = c(28,35,21,36),
  c130 = c(180,235,175,238)
)

# Adding dbh
data <- add_c130_dbh(data)

# Computes volumes
data <- algan_vta_vc22(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_SYLVESTRIS","ABIES_ALBA"),
  htot = c(20,25,18,35),
  c130 = c(180,235,175,238)
)

# Computes volumes
data <- dagnelie_vc22_2(data)

# Computes biomass
data <- biomass_calc(data)


## ----echo=FALSE---------------------------------------------------------------
scroll_box(
  gembloux_kable(data),
  width = "100%",
  height = "200px"
)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_SYLVESTRIS","ABIES_ALBA"),
  htot = c(20,25,18,35),
  c130 = c(180,235,175,238)
)

# Computes volumes
data <- vallet_vta(data)

# Computes biomass
data <- biomass_calc(data)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## -----------------------------------------------------------------------------
# Data creation
data <- data.frame(
  species_code = c("ABIES_ALBA","QUERCUS_ROBUR","PINUS_SYLVESTRIS","ABIES_ALBA"),
  c130 = c(180,235,175,238)
)

# Type of path needed (must be replaced by a real path)
# filepath <- "file/path/example.csv"

## ----echo=FALSE---------------------------------------------------------------
# For the example (do not run)
filepath <- tempfile(fileext = ".csv")

## ----message=FALSE------------------------------------------------------------
# Computes volumes and export csv
data <- dagnelie_vc22_1(data, output= filepath)


## ----echo=FALSE---------------------------------------------------------------
gembloux_kable(data)

## ----echo=FALSE---------------------------------------------------------------
set.seed(123)  # reproductible

n <- 150

# c130 entre 50 et 200
c130 <- runif(n, min = 50, max = 200)

# hauteur proportionnelle (ex: htot ≈ 0.25 * c130 + bruit aléatoire)
htot <- 0.25 * c130 + rnorm(n, mean = 0, sd = 3)

# bornes réalistes (hauteur min 5 m, max 45 m)
htot <- pmax(5, pmin(htot, 45))

# quelques espèces possibles
species_list <- c(
  "PINUS_SYLVESTRIS", "PICEA_ABIES", "QUERCUS_ROBUR",
  "FAGUS_SYLVATICA", "BETULA_SP"
)

species_code <- sample(species_list, n, replace = TRUE)

# construction
df <- data.frame(
  c130 = round(c130, 1),
  htot = round(htot, 1),
  species_code = species_code
)

## ----echo=TRUE----------------------------------------------------------------
# Type of path needed (must be replaced by a real path)
# filepath <- "my/file/path.csv"

## ----echo=FALSE---------------------------------------------------------------
# For the example
filepath <- tempfile(fileext = ".csv")

# Calcul des volumes
data <- dagnelie_vc22_1(df)

## ----echo=TRUE, message=FALSE, warning=FALSE, fig.width=6, fig.height=6, fig.align="center"----
# Plot + export CSV
res <- plot_by_class(data, volume_col = "dagnelie_vc22_1",
                    output = filepath)

# Affichage du plot (important !)
res$plot

## ----echo=FALSE---------------------------------------------------------------
# read the exported csv
csv_data <- read.csv2(filepath)

# display it with gembloux style
gembloux_kable(csv_data, caption = "CSV Output Preview")

