% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RVcomparison.R
\name{RVcomparison}
\alias{RVcomparison}
\title{Compare Escoufier RV coefficient between groups}
\usage{
RVcomparison(Block1, Block2, group, perm = 999, center = TRUE)
}
\arguments{
\item{Block1, Block2}{Matrices or data frames
containing each block of variables
(observations in rows, variables in columns).}

\item{group}{factor or vector indicating group membership for observations.}

\item{perm}{number of permutations for the test}

\item{center}{whether the groups should be mean-centered prior to the test}
}
\value{
A data frame with one row per pairwise comparison and the
  following columns:
 \describe{
  \item{group1}{Name of the first group in the comparison}
  \item{group2}{Name of the second group in the comparison}
  \item{Observed_RV_group1}{Observed Escoufier RV for the first
    group in the comparison}
  \item{Observed_RV_group2}{Observed Escoufier RV for the second
    group in the comparison}
  \item{Absolute_difference_in_RV}{Absolute difference in the
    observed Escoufier RV between the two groups}
  \item{p_value}{p value of the permutation test}
}
For multiple groups, the data frame includes additional columns
identifying the groups being compared.
}
\description{
Performs permutation tests for the difference in
Escoufier RV between groups. For multiple groups, performs
pairwise comparisons between all pairs of groups.
}
\details{
This function is one of the solutions proposed by Fruciano et al.
2013 to deal with the fact that values of Escoufier RV coefficient
(Escoufier 1973), which is routinely used to quantify the levels
of association between multivariate blocks of variables (landmark
coordinates in the case of morphometric data, but it might be any
multivariate dataset)
are not comparable across groups with different number of
observations/individuals (Smilde et al. 2009; Fruciano et al. 2013).
The solution is a permutation test. This test was originally
implemented by Adriano Franchini in the Java program RVcomparator,
of which this function is an updated and improved version
}
\section{Notice}{

The values of RV for each of the groups the function outputs
are the observed ones, and can be reported but their value should
not be compared. If one wants to obtain comparable values
one solution (Fruciano et al 2013) is to obtain rarefied estimates,
which can be done with the function RVrarefied in this package
}

\section{Citation}{

If you use this function please cite both
Fruciano et al. 2013 (for using the rarefaction procedure)
and Escoufier 1973 (because the procedure is based on Escoufier RV)
}

\examples{

library(MASS)
set.seed(123)

# Create sample data with different correlation structures
S1 = diag(50)  # Uncorrelated variables for group 1
S2 = diag(50)
S2[11:50, 11:50] = 0.3  # Some correlation in second block for group 2
S2 = (S2 + t(S2)) / 2  # Ensure symmetry
diag(S2) = 1

# Generate data for two groups
A = mvrnorm(30, mu = rep(0, 50), Sigma = S1)
B = mvrnorm(40, mu = rep(0, 50), Sigma = S2)

# Combine data and create group labels
combined_data1 = A[, 1:20]
combined_data2 = A[, 21:50]
combined_data1 = rbind(combined_data1, B[, 1:20])
combined_data2 = rbind(combined_data2, B[, 21:50])
groups = c(rep("GroupA", 30), rep("GroupB", 40))

# Perform RV comparison
result = RVcomparison(combined_data1, combined_data2,
                      group = groups, perm = 99)
print(result)

# Example with three groups for pairwise comparisons
C = mvrnorm(25, mu = rep(0, 50), Sigma = diag(50))
combined_data1_3grp = rbind(combined_data1, C[, 1:20])
combined_data2_3grp = rbind(combined_data2, C[, 21:50])
groups_3 = c(groups, rep("GroupC", 25))

result_3grp = RVcomparison(combined_data1_3grp, combined_data2_3grp, 
                          group = groups_3, perm = 99)
print(result_3grp)

}
\references{
Escoufier Y. 1973. Le Traitement des Variables
  Vectorielles. Biometrics 29:751-760.

Fruciano C, Franchini P, Meyer A. 2013.
  Resampling-Based Approaches to Study Variation in Morphological
  Modularity. PLoS ONE 8:e69376.

Smilde AK, Kiers HA, Bijlsma S, Rubingh CM, van Erk MJ.
  2009. Matrix correlations for high-dimensional data: the modified
  RV-coefficient. Bioinformatics 25:401-405.
}
\seealso{
\code{\link{EscoufierRV}}

\code{\link{RVrarefied}}
}
