\name{plotHSMMParameters}
\alias{plotHSMMParameters}
\title{Plot Hidden Semi-Markov Model Parameters Over Time}
\description{
Plots state-dependent observation parameters and dwell-time parameters of a fitted Hidden Semi-Markov Model (HSMM) over time.
Confidence intervals can be included, with optional overlay of observed data for comparison.
}
\usage{
plotHSMMParameters(x, HSMM, obsdist, dwelldist, confint_result = NULL,
                   level = 0.95, B = 100, M = NA, include_dwell = FALSE,
                   shift = FALSE, time_structure = NULL,
                   plot_title = "HSMM Parameters Over Time",
                   overlay_data = NULL, overlay_label = "Data",
                   colors = c("black", "red", "blue", "darkgreen"),
                   save_plot = FALSE, filename = NULL,
                   width = 12, height = 8, dpi = 300,
                   verbose = TRUE, seed = NULL)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{HSMM}{A fitted HSMM object (typically the output from \code{\link{findmleHSMM}}),
containing estimated transition probabilities \code{Pi}, initial probabilities \code{delta},
state-dependent observation parameters \code{observationparameters}, and dwell-time parameters \code{dwellparameters}.}

\item{obsdist}{Character string. Observation distribution. Supported: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{dwelldist}{Character string. Dwell-time distribution. Supported: "pois", "nbinom", "geom", "betabinom".}

\item{confint_result}{Optional result from \code{\link{confintHSMM}}. If not provided, confidence intervals are computed internally.}

\item{level}{Numeric between 0 and 1. Confidence level for parameter intervals. Default is 0.95.}

\item{B}{Integer. Number of bootstrap replicates for confidence intervals if not supplied. Default is 100.}

\item{M}{Integer. Maximum dwell time used in decoding. Defaults to \code{min(1000, length(x))} if \code{NA}.}

\item{include_dwell}{Logical. If \code{TRUE}, the dwell parameters will also be plotted through time. Default is \code{FALSE}.}

\item{shift}{Logical. If \code{TRUE}, fits shifted dwell-time distributions. Default is \code{FALSE}.}

\item{time_structure}{Optional list specifying the time scale.
Must include \code{unit}, \code{observations_per_unit}, and optionally \code{start_point}, \code{end_point}.
Supports calendar units ("day", "year", "hour") or custom scaling.}

\item{plot_title}{Character string. Title for the full plot layout. Default is "HSMM Parameters Over Time".}

\item{overlay_data}{Optional numeric vector of length \code{length(x)}. External data to overlay on parameter plots.}

\item{overlay_label}{Character string. Label for overlay axis. Default is "Data".}

\item{colors}{Character vector of plot colors. Default is \code{c("black", "red", "blue", "darkgreen")}.}

\item{save_plot}{Logical. If \code{TRUE}, the plot is saved to a file. Default is \code{FALSE}.}

\item{filename}{Character string. File path for saving the plot. Required if \code{save_plot = TRUE}.}

\item{width}{Numeric. Width of saved plot in inches. Default is 12.}

\item{height}{Numeric. Height of saved plot in inches. Default is 8.}

\item{dpi}{Numeric. Resolution (dots per inch) for saved plot. Default is 300.}

\item{verbose}{Logical. If \code{TRUE}, progress messages are printed to the console. Default is \code{TRUE}.}

\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
}
\value{
Invisibly returns a list containing:
\item{obs_param_series}{List of observation parameter time series for each parameter.}
\item{obs_ci_series}{List of confidence interval time series (lower and upper bounds) for each observation parameter.}
\item{dwell_param_series}{List of dwell parameter time series for each parameter (if \code{include_dwell = TRUE}).}
\item{dwell_ci_series}{List of confidence interval time series for each dwell parameter (if \code{include_dwell = TRUE}).}
\item{time_info}{List containing time axis information and labels.}
\item{decoded_states}{Vector of decoded hidden states for each time point.}

The function also generates multi-panel plots of parameter trajectories with confidence intervals and optional overlays.
}

\details{
The most likely hidden state sequence is decoded using \code{\link{globaldecodeHSMM}}.
For each decoded state, observation and dwell-time parameters are extracted and plotted over time.
If \code{confint_result} is not provided, bootstrap intervals are computed via \code{\link{confintHSMM}}.
Plots include observation parameter trajectories, dwell-time parameter trajectories (if \code{include_dwell = TRUE}), and optional overlay data on secondary axes.
Time scaling is configurable using \code{time_structure}.
When \code{verbose = TRUE}, progress messages are displayed during confidence interval computation.
If \code{save_plot = TRUE}, the plot is saved as a PNG file to the specified \code{filename} with the given dimensions and resolution.
}
\examples{
set.seed(42)
J <- 2
Pi <- matrix(c(0.0, 1.0,
               1.0, 0.0), nrow = J, byrow = TRUE)

obspar <- list(mean = c(0, 3), sd = c(1, 1.2))
dwellpar <- list(lambda = c(5, 8))

sim <- generateHSMM(n = 200, J = J,
                    obsdist = "norm", dwelldist = "pois",
                    obspar = obspar, dwellpar = dwellpar,
                    Pi = Pi)

HSMM_fit <- findmleHSMM(x = sim$x, J = J, M = 100,
                        obsdist = "norm", dwelldist = "pois",
                        obspar = obspar, dwellpar = dwellpar,
                        Pi = Pi)

overlay_series <- sim$x
time_struct <- list(unit = "day", observations_per_unit = 10, start_point = 1)

\donttest{
result <- plotHSMMParameters(x = sim$x, HSMM = HSMM_fit,
                             obsdist = "norm", dwelldist = "pois",
                             overlay_data = overlay_series,
                             overlay_label = "Observed values",
                             time_structure = time_struct,
                             plot_title = "HSMM Parameters with Overlay")

result_silent <- plotHSMMParameters(x = sim$x, HSMM = HSMM_fit,
                                    obsdist = "norm", dwelldist = "pois",
                                    include_dwell = TRUE,
                                    overlay_data = overlay_series,
                                    time_structure = time_struct,
                                    verbose = FALSE)
}
}
\seealso{
\code{\link{findmleHSMM}}, for fitting HSMMs.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{globaldecodeHSMM}}, for decoding the most likely state sequence.
\code{\link{confintHSMM}}, for bootstrap confidence intervals of HSMM parameters.
}
\author{Aimee Cody}

