% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metaODIN.R
\name{meta_sim}
\alias{meta_sim}
\title{Metapopulation Respiratory Virus Model Simulator}
\usage{
meta_sim(
  N_pop,
  ts,
  tv,
  S0,
  I0,
  P0,
  R0,
  H0 = rep(0, N_pop),
  D0 = rep(0, N_pop),
  Ia0 = rep(0, N_pop),
  Ip0 = rep(0, N_pop),
  E0 = rep(0, N_pop),
  V0 = rep(0, N_pop),
  m_weekday_day,
  m_weekday_night,
  m_weekend_day,
  m_weekend_night,
  start_day = 0,
  delta_t,
  vac_mat,
  dv,
  de,
  pea,
  dp,
  da,
  ds,
  psr,
  dh,
  phr,
  dr,
  ve,
  nsteps,
  is.stoch = FALSE,
  seed = NULL,
  do_chk = FALSE,
  chk_time_steps = NULL,
  chk_file_names = NULL
)
}
\arguments{
\item{N_pop}{Integer. Number of demographic subpopulations in the model}

\item{ts}{Numeric vector or scalar. Transmission rate for symptomatic individuals
in susceptible population. If scalar, applied to all subpopulations}

\item{tv}{Numeric vector or scalar. Transmission rate for symptomatic individuals
in vaccinated population. If scalar, applied to all subpopulations}

\item{S0}{Numeric vector of length N_pop. Initial number of susceptible individuals
in each subpopulation}

\item{I0}{Numeric vector of length N_pop. Initial number of symptomatic infected
individuals in each subpopulation}

\item{P0}{Numeric vector of length N_pop. Total population sizes for each subpopulation}

\item{R0}{Numeric vector of length N_pop. Initial number of recovered individuals
in each subpopulation}

\item{H0}{Numeric vector of length N_pop. Initial number of hospitalized individuals
in each subpopulation (default: rep(0, N_pop))}

\item{D0}{Numeric vector of length N_pop. Initial number of deceased individuals
in each subpopulation (default: rep(0, N_pop))}

\item{Ia0}{Numeric vector of length N_pop. Initial number of asymptomatic infected
individuals in each subpopulation (default: rep(0, N_pop))}

\item{Ip0}{Numeric vector of length N_pop. Initial number of presymptomatic infected
individuals in each subpopulation (default: rep(0, N_pop))}

\item{E0}{Numeric vector of length N_pop. Initial number of exposed individuals
in each subpopulation (default: rep(0, N_pop))}

\item{V0}{Numeric vector of length N_pop. Initial number of vaccinated individuals
in each subpopulation}

\item{m_weekday_day}{Numeric matrix (N_pop × N_pop). Contact mixing matrix for
weekday daytime (6 AM - 6 PM) interactions}

\item{m_weekday_night}{Numeric matrix (N_pop × N_pop). Contact mixing matrix for
weekday nighttime (6 PM - 6 AM) interactions}

\item{m_weekend_day}{Numeric matrix (N_pop × N_pop). Contact mixing matrix for
weekend daytime (6 AM - 6 PM) interactions}

\item{m_weekend_night}{Numeric matrix (N_pop × N_pop). Contact mixing matrix for
weekend nighttime (6 PM - 6 AM) interactions}

\item{start_day}{Start day of the week expressed as an integer value between 0 and 6, 0 being
Monday. Default simulation start day is Monday.}

\item{delta_t}{Positive numeric. Discrete time increment in days (typically 0.5)}

\item{vac_mat}{Numeric matrix. Vaccination schedule with dimensions (nsteps × (1 + N_pop)).
First column contains time indices, remaining columns contain vaccination counts
for each subpopulation at each time step}

\item{dv}{Numeric vector or scalar. Mean duration (days) in vaccinated state before
immunity waning. If scalar, applied to all subpopulations}

\item{de}{Numeric vector or scalar. Mean duration (days) in exposed state.
If scalar, applied to all subpopulations}

\item{pea}{Numeric vector or scalar. Proportion of exposed individuals becoming
asymptomatic infectious (vs. presymptomatic), values between 0 and 1. If scalar, applied to all subpopulations.
If scalar, applied to all subpopulations}

\item{dp}{Numeric vector or scalar. Mean duration (days) in presymptomatic
infectious state. If scalar, applied to all subpopulations}

\item{da}{Numeric vector or scalar. Mean duration (days) in asymptomatic
infectious state. If scalar, applied to all subpopulations}

\item{ds}{Numeric vector or scalar. Mean duration (days) in symptomatic
infectious state. If scalar, applied to all subpopulations}

\item{psr}{Numeric vector or scalar. Proportion of symptomatic individuals
recovering directly (vs. hospitalization), values between 0 and 1. If scalar, applied to all subpopulations.
If scalar, applied to all subpopulations}

\item{dh}{Numeric vector or scalar. Mean duration (days) in hospitalized state.
If scalar, applied to all subpopulations}

\item{phr}{Numeric vector or scalar. Proportion of hospitalized individuals
recovering (vs. death). , values between 0 and 1. If scalar, applied to all subpopulations.}

\item{dr}{Numeric vector or scalar. Mean duration (days) of immunity in
recovered state. If scalar, applied to all subpopulations}

\item{ve}{Numeric vector or scalar. Vaccine effectiveness (proportion)
, values between 0 and 1. If scalar, applied to all subpopulations}

\item{nsteps}{Integer. Total number of discrete time evolution steps in simulation}

\item{is.stoch}{Logical. Whether to run stochastic simulation (TRUE) or
deterministic simulation (FALSE). Default: FALSE}

\item{seed}{Integer or NULL. Random seed for reproducibility. Only used when
is.stoch = TRUE. Default: NULL}

\item{do_chk}{Logical. Whether to save model checkpoint at simulation end.
Default: FALSE}

\item{chk_time_steps}{Integer vector or NULL. Time steps at which to save checkpoints.}

\item{chk_file_names}{List of character vectors or NULL. File names for checkpoints.
Each element of the list corresponds to a time step in \code{chk_time_steps}.}
}
\value{
Returns a data.table with the following structure:
\describe{
\item{step}{Integer time step index (0 to nsteps)}
\item{time}{Continuous simulation time (step × delta_t)}
\item{disease_state}{Character vector of compartment names}
\item{population_id}{Character vector of subpopulation identifiers}
\item{value}{Numeric values representing population counts in each compartment}
}

Available disease states in output:
\itemize{
\item Core compartments: S, E, I_presymp, I_asymp, I_symp, H, R, D, V, P
\item Derived outputs: I_all (total infectious), cum_V (cumulative vaccinations)
\item Transition flows: n_SE, n_EI, n_HR, n_HD, etc. (new infections, hospitalizations, deaths)
\item Debug outputs: p_SE, p_VE, I_eff (probabilities and effective populations)
}
}
\description{
The core simulation engine that implements a stochastic compartmental SEIRD
(Susceptible-Exposed-Infected-Recovered-Dead) model for respiratory virus
epidemics across multiple demographic subpopulations. The function compiles
and executes an ODIN-based differential equation model with time-varying
contact patterns, vaccination dynamics, and complex disease progression pathways.
}
\details{
The model implements a metapopulation epidemiological framework with the following features:

\strong{Compartmental Structure:}
\itemize{
\item \strong{S}: Susceptible individuals
\item \strong{E}: Exposed (incubating) individuals
\item \strong{I_presymp}: Presymptomatic infectious individuals
\item \strong{I_asymp}: Asymptomatic infectious individuals
\item \strong{I_symp}: Symptomatic infectious individuals
\item \strong{H}: Hospitalized individuals
\item \strong{R}: Recovered individuals
\item \strong{D}: Deceased individuals
\item \strong{V}: Vaccinated individuals
\item \strong{P}: Total living population (excludes deaths)
}

\strong{Disease Progression Pathways:}
\enumerate{
\item \strong{S → E}: Exposure through contact with infectious individuals
\item \strong{E → I_asymp/I_presymp}: Progression to infectious states (proportion pea)
\item \strong{I_presymp → I_symp}: Development of symptoms
\item \strong{I_asymp → R}: Direct recovery from asymptomatic state
\item \strong{I_symp → R/H}: Recovery or hospitalization (proportion psr)
\item \strong{H → R/D}: Hospital discharge or death (proportion phr)
\item \strong{R → S}: Loss of immunity
\item \strong{S → V}: Vaccination
\item \strong{V → S/E}: Vaccine waning or breakthrough infection
}

\strong{Mixing Patterns:}
Contact patterns vary by:
\itemize{
\item Day of week: Weekday vs. weekend patterns
\item Time of day: Day (6 AM - 6 PM) vs. night (6 PM - 6 AM) patterns
\item Each pattern specified by N_pop × N_pop contact matrix
}

\strong{Force of Infection:}
Transmission occurs through contact between susceptible/vaccinated individuals and
all infectious compartments (I_presymp + I_asymp + I_symp), modified by:
\itemize{
\item Population-specific transmission rates (ts, tv)
\item Time-varying contact patterns
\item Vaccine effectiveness for breakthrough infections
}

\strong{Stochastic vs. Deterministic Mode:}
\itemize{
\item \strong{Deterministic}: Uses exact differential equations
\item \strong{Stochastic}: Adds demographic stochasticity via binomial draws
}

\strong{Vaccination Implementation:}
Vaccination is implemented as time-varying input with:
\itemize{
\item Scheduled vaccination counts per time step and subpopulation
\item Vaccine effectiveness reducing infection probability
\item Waning immunity returning individuals to susceptible state
}
}
\section{Parameter Scaling}{

All duration parameters are automatically converted to rates (1/duration).
Scalar parameters are automatically expanded to vectors of length N_pop.
This allows flexible specification of homogeneous or heterogeneous parameters.
}

\section{Checkpointing}{

When do_chk = TRUE, the function saves a checkpoint file containing:
\itemize{
\item Final compartment states for simulation continuation
\item All model parameters for reproducibility
\item Vaccination schedule data
\item Population structure information
}
}

\examples{
\donttest{
options(odin.verbose = FALSE)
# Basic deterministic simulation
N_pop <- 2
nsteps <- 400

# Initialize populations
S0 <- rep(1000, N_pop)
I0 <- rep(10, N_pop)
P0 <- S0 + I0
R0 <- rep(0, N_pop)

# Contact matrices (simplified - identity matrices)
contact_matrix <- diag(N_pop)

# Basic vaccination schedule (10\% vaccination)
vac_mat <- matrix(0, nrow = nsteps + 1, ncol = N_pop + 1)
vac_mat[, 1] <- 0:nsteps
vac_mat[1, 1 + (1:N_pop)] <- P0 * 0.1

# Run simulation
results <- meta_sim(
  N_pop = N_pop,
  ts = 0.5,
  tv = 0.1,
  S0 = S0,
  I0 = I0,
  P0 = P0,
  R0 = R0,
  m_weekday_day = contact_matrix,
  m_weekday_night = contact_matrix,
  m_weekend_day = contact_matrix,
  m_weekend_night = contact_matrix,
  delta_t = 0.5,
  vac_mat = vac_mat,
  dv = 365,
  de = 3,
  pea = 0.3,
  dp = 2,
  da = 7,
  ds = 7,
  psr = 0.95,
  dh = 10,
  phr = 0.9,
  dr = 180,
  ve = 0.8,
  nsteps = nsteps,
  is.stoch = FALSE
)
}


}
\references{
\itemize{
\item ODIN package: \url{https://mrc-ide.github.io/odin/}
\item Fadikar, A., et al. "Developing and deploying a use-inspired metapopulation modeling framework for detailed tracking of stratified health outcomes"
}
}
\seealso{
\code{\link{metaRVM}} for high-level simulation interface with configuration files
\code{\link{parse_config}} for configuration file processing
\code{\link{format_metarvm_output}} for output formatting with demographics
}
\author{
Arindam Fadikar, Charles Macal, Ignacio Martinez-Moyano, Jonathan Ozik
}
