% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixpflik.R
\name{mixpf}
\alias{mixpf}
\title{Profile Likelihood Method for Normal Mixture with Unequal Variance}
\usage{
mixpf(x, k = 0.5, C = 2, nstart = 20)
}
\arguments{
\item{x}{a vector of observations.}

\item{k}{ratio of the smallest variance to the largest variance. Default is 0.5.}

\item{C}{number of mixture components. Default is 2.}

\item{nstart}{number of initializations to try. Default is 20.}
}
\value{
A list containing the following elements:
\item{mu}{vector of estimated component means.}
\item{sigma}{vector of estimated component standard deviations.}
\item{pi}{vector of estimated mixing proportions.}
\item{lik}{final likelihood.}
}
\description{
`mixpf' is used to estimate the following \eqn{C}-component univariate normal mixture model,
using the profile likelihood method (Yao, 2010), with the assumption that the ratio of
the smallest variance to the largest variance is \eqn{k}:
\deqn{f(x;\boldsymbol{\theta}) = \sum_{j=1}^C\pi_j\phi(x;\mu_j,\sigma_j^2),}
where \eqn{\boldsymbol{\theta}=(\pi_1,\mu_1,\sigma_1,..,\pi_{C},\mu_C,\sigma_C)^{\top}}
is the parameter to estimate, \eqn{\phi(\cdot;\mu,\sigma^2)} is the normal density with a
mean of \eqn{\mu} and a standard deviation of \eqn{\sigma}, and \eqn{\pi}'s are mixing
proportions that sum up to 1.
Once the results are obtained, one can also find the maximum likelihood estimate (MLE) of \eqn{k} by
plotting the likelihood vs. \eqn{k} for different \eqn{k} values and finding the maximum
interior mode in the likelihood. See examples below.
}
\examples{
set.seed(4)
n = 100
u = runif(n, 0, 1)
x2 = (u <= 0.3) * rnorm(n, 0, 0.5) + (u > 0.3) * rnorm(n, 1.5, 1)

# please set ngrid to 200 to get a smooth likelihood curve
ngrid = 5
grid = seq(from = 0.01, to = 1, length = ngrid)
likelihood = numeric()
for(i in 1:ngrid){
  k = grid[i]
  est = mixpf(x2, k)
  lh = est$lik
  likelihood[i] = lh
}

# visualize likelihood to find the best k
plot(grid, likelihood, type = "l", lty = 2, xlab = "k", ylab = "profile log-likelihood")
}
\references{
Yao, W. (2010). A profile likelihood method for normal mixture with unequal variance.
Journal of Statistical Planning and Inference, 140(7), 2089-2098.
}
