# Clean environment and load libraries
rm(list = ls())  # Clear environment to avoid conflicts

# Source files
library(MixStable)

# 📊 Generate synthetic data using explicit package reference
X <- stabledist::rstable(1200, alpha = 1.2, beta = 0.5, gamma = 1, delta = 3, pm = 1)
u_vals <- seq(0.1, 1, length.out = 10)

# Add some diagnostic information
cat("Data summary:\n")
cat("Length:", length(X), "\n")
cat("Range:", range(X), "\n")
cat("Mean:", mean(X), "SD:", sd(X), "\n")
cat("u_vals:", u_vals, "\n\n")

# 🧮 Estimate parameters with error handling
safe_estimate <- function(func, data, u, name) {
  cat("Running", name, "...\n")
  result <- tryCatch({
    func(data, u)
  }, error = function(e) {
    cat("Error in", name, ":", e$message, "\n")
    return(NULL)
  }, warning = function(w) {
    cat("Warning in", name, ":", w$message, "\n")
    suppressWarnings(func(data, u))
  })
  return(result)
}

# Run estimations with error handling
result_CDF <- safe_estimate(CDF, X, u_vals, "CDF")
result_ecf_regression <- safe_estimate(ecf_regression, X, u_vals, "ECF Regression")
result_robust_ecf_regression <- safe_estimate(robust_ecf_regression, X, u_vals, "Robust ECF Regression")
result_fit_stable_ecf <- safe_estimate(fit_stable_ecf, X, u_vals, "Fit Stable ECF")
result_ecf <- safe_estimate(estimate_stable_recursive_ecf, X, u_vals, "Recursive ECF")
result_kernel <- safe_estimate(estimate_stable_kernel_ecf, X, u_vals, "Kernel ECF")
result_weighted <- safe_estimate(estimate_stable_weighted_ols, X, u_vals, "Weighted OLS")
result_cdf_ <- safe_estimate(estimate_stable_from_cdf, X, u_vals, "CDF-based")

# 📋 Display results
display_result <- function(result, name) {
  cat("\n=== ", name, " Results ===\n")
  if (is.null(result)) {
    cat("Failed to compute\n")
  } else {
    print(result)
  }
}

display_result(result_CDF, "CDF")
display_result(result_ecf_regression, "ECF Regression")
display_result(result_robust_ecf_regression, "Robust ECF Regression")
display_result(result_fit_stable_ecf, "Fit Stable ECF")
display_result(result_ecf, "Recursive ECF")
display_result(result_kernel, "Kernel ECF")
display_result(result_weighted, "Weighted OLS")
display_result(result_cdf_, "CDF-based")

# Compare with true parameters
cat("\n=== True Parameters ===\n")
cat("alpha: 1.2, beta: 0.5, gamma: 1, delta: 3\n")
