#' Calculate Node and Global Metrics for Trait Networks
#'
#' This function computes various node and global metrics for a trait network graph.
#'
#' @param graph An igraph object representing the trait network, typically generated by the `PTN` function.
#'
#' @return A list containing two data frames:
#'   \item{node}{A data frame with node-level metrics including degree, closeness, betweenness, and local clustering coefficient.}
#'   \item{global}{A data frame with global metrics including edge density, diameter, average path length, average clustering coefficient, and modularity.}
#'
#' @references
#' 1. He, N., Li, Y., Liu, C., et al. (2020). Plant trait networks: improved resolution of the dimensionality of adaptation.
#' Trends in Ecology & Evolution, 35(10), 908-918.
#' 2. Li, Y., Liu, C., Sack, L., Xu, L., Li, M., Zhang, J., & He, N. (2022). Leaf trait network architecture shifts with
#' species‐richness and climate across forests at continental scale. Ecology Letters, 25(6), 1442-1457.
#'
#' @examples
#' data(PFF)
#' rownames(PFF) <- PFF$species
#' PFF_traits <- PFF[, c("Height", "Leaf_area","LDMC","SLA","SRL","SeedMass","FltDate",
#'                       "FltDur","Leaf_Cmass","Leaf_Nmass","Leaf_CN","Leaf_Pmass",
#'                       "Leaf_NP","Leaf_CP","Root_Cmass","Root_Nmass","Root_CN")]
#' PFF_traits <- na.omit(PFF_traits)
#' head(PFF_traits)
#' ptn_result <- PTN(traits_matrix = PFF_traits, rThres = 0.2, pThres = 0.05)
#' PTN_metrics(ptn_result)
#'
#' data(PFF_tree)
#' ptn_phylo_result <- PTN(traits_matrix = PFF_traits,
#'                       rThres = 0.2,
#'                       pThres = 0.05,
#'                       method = "pearson",
#'                       phylo_correction = TRUE,
#'                       phylo_tree = PFF_tree)
#' PTN_metrics(ptn_phylo_result)
#'
#' @importFrom igraph degree closeness betweenness transitivity edge_density diameter mean_distance cluster_fast_greedy modularity membership
#' @export
PTN_metrics <- function(graph) {
  # Node-level metrics
  degree <- igraph::degree(graph, mode = "all")
  closeness <- igraph::closeness(graph, mode = "all", weights = NA)
  betweenness <- igraph::betweenness(graph, directed = FALSE, weights = NA)
  local_clustering <- igraph::transitivity(graph, type = "local", weights = NA)
  # Set the clustering coefficient of nodes with degree 0 or 1 to 0
  local_clustering[is.nan(local_clustering) | degree <= 1] <- 0
  # Creating node-level metrics dataframe
  node_metrics <- data.frame(
    degree = degree,
    closeness = closeness,
    betweenness = betweenness,
    clustering_coefficient = local_clustering
  )
  # Global metrics
  edge_density <- igraph::edge_density(graph, loops = FALSE)
  diameter <- igraph::diameter(graph, directed = FALSE, weights = NA)
  avg_path_length <- igraph::mean_distance(graph, directed = FALSE)
  avg_clustering <- igraph::transitivity(graph, type = "global", weights = NA)
  fc <- suppressWarnings(igraph::cluster_edge_betweenness(graph, weights = NA))
  modularity <- igraph::modularity(fc)
  # Creating global metrics dataframe
  global_metrics <- data.frame(
    edge_density = edge_density,
    diameter = diameter,
    avg_path_length = avg_path_length,
    avg_clustering_coefficient = avg_clustering,
    modularity = modularity
  )
  # Return a list containing both node and global metrics
  return(list(
    node = node_metrics,
    global = global_metrics
  ))
}
