# HCGenRayleigh.R
#' Half-Cauchy Generalized Rayleigh Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy Generalized Rayleigh distribution.
#'
#' The Half-Cauchy Generalized Rayleigh distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Half-Cauchy Generalized Rayleigh distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \quad 1 - \frac{2}{\pi }\arctan \left\{ { - \frac{\alpha }{\theta }
#' \log \left \{ {1 - {e^{ - {{\left( {\lambda x} \right)}^2}}}} \right\}} \right\} \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dhc.gen.rayleigh()} — Density function
#'   \item \code{phc.gen.rayleigh()} — Distribution function
#'   \item \code{qhc.gen.rayleigh()} — Quantile function
#'   \item \code{rhc.gen.rayleigh()} — Random generation
#'   \item \code{hhc.gen.rayleigh()} — Hazard function
#' }
#'
#' @usage
#' dhc.gen.rayleigh(x, alpha, lambda, theta, log = FALSE)
#' phc.gen.rayleigh(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.gen.rayleigh(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.gen.rayleigh(n, alpha, lambda, theta)
#' hhc.gen.rayleigh(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.gen.rayleigh}: numeric vector of (log-)densities
#'   \item \code{phc.gen.rayleigh}: numeric vector of probabilities
#'   \item \code{qhc.gen.rayleigh}: numeric vector of quantiles
#'   \item \code{rhc.gen.rayleigh}: numeric vector of random variates
#'   \item \code{hhc.gen.rayleigh}: numeric vector of hazard values
#' }
#'
#' @references 
#' Sapkota, L.P., & Kumar, V. (2023). 
#' Half-Cauchy Generalized Rayleigh : Theory and Applications.\emph{South East Asian 
#' J. Math. & Math. Sc.}, \bold{19(1)}, 335--360. \doi{10.56827/SEAJMMS.2023.1901.27}
#'
#' Shrestha, S.K., & Kumar, V. (2014). 
#' Bayesian Analysis for the Generalized Rayleigh Distribution. 
#' \emph{International Journal of Statistika and Mathematika}, \bold{9(3)}, 118--131.
#'
#' Kundu, D., & Raqab, M.Z. (2005). 
#' Generalized Rayleigh Distribution: Different Methods of Estimation. 
#' \emph{Computational Statistics and Data Analysis}, \bold{49}, 187--200.
#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dhc.gen.rayleigh(x, 2.0, 0.5, 0.1)
#' phc.gen.rayleigh(x, 2.0, 0.5, 0.1)
#' qhc.gen.rayleigh(0.5, 2.0, 0.5, 0.1)
#' rhc.gen.rayleigh(10, 2.0, 0.5, 0.1)
#' hhc.gen.rayleigh(x, 2.0, 0.5, 0.1)
#' 
#' # Data
#' x <- stress66
#' # ML estimates
#' params = list(alpha=1.4585, lambda=0.5300, theta=0.1655)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.gen.rayleigh, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.gen.rayleigh, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dhc.gen.rayleigh, pfun=phc.gen.rayleigh, plot=FALSE)
#' print.gofic(out)
#' 
#' @name HCGenRayleigh
#' @aliases dhc.gen.rayleigh phc.gen.rayleigh qhc.gen.rayleigh rhc.gen.rayleigh hhc.gen.rayleigh
#' @export
NULL

#' @rdname HCGenRayleigh
#' @usage NULL
#' @export   
dhc.gen.rayleigh <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- (lambda * x[valid])^2	
	z <- 1 - exp(-u)	
    pdf[valid] <- (4/pi)*(alpha*lambda*lambda/theta)*x[valid]*exp(-u) *(1/z) * (1/(1+(-alpha*log(z)/theta)^2))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCGenRayleigh
#' @usage NULL
#' @export   
phc.gen.rayleigh <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -alpha * log(1- exp(-lambda * lambda * q[valid] * q[valid])) / theta	
    cdf[valid] <- 1- (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCGenRayleigh
#' @usage NULL
#' @export   
qhc.gen.rayleigh <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	t <- exp(-theta * tan((1 - p[valid]) * pi * 0.5)) 	
    qtl[valid] <- (1/lambda) * (-log(1 - t ^(1/alpha))) ^ 0.5	     
    }    
    return(qtl)  
 } 

#' @rdname HCGenRayleigh
#' @usage NULL
#' @export   
rhc.gen.rayleigh <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.gen.rayleigh(u, alpha, lambda, theta)
    return(q)
}

#' @rdname HCGenRayleigh
#' @usage NULL
#' @export   
hhc.gen.rayleigh <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dhc.gen.rayleigh(x, alpha, lambda, theta) 
	surv <- 1.0 - phc.gen.rayleigh(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 



