# HCGompertz.R
#' Half-Cauchy Gompertz Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy Gompertz distribution.
#'
#' The Half-Cauchy Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Half-Cauchy Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \quad \frac{2}{\pi }\arctan \left\{ { - \frac{\lambda }{{\alpha \theta }}
#' \left( {1 - {e^{\alpha x}}} \right)} \right\} \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dhc.gpz()} — Density function
#'   \item \code{phc.gpz()} — Distribution function
#'   \item \code{qhc.gpz()} — Quantile function
#'   \item \code{rhc.gpz()} — Random generation
#'   \item \code{hhc.gpz()} — Hazard function
#' }
#'
#' @usage
#' dhc.gpz(x, alpha, lambda, theta, log = FALSE)
#' phc.gpz(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.gpz(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.gpz(n, alpha, lambda, theta)
#' hhc.gpz(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.gpz}: numeric vector of (log-)densities
#'   \item \code{phc.gpz}: numeric vector of probabilities
#'   \item \code{qhc.gpz}: numeric vector of quantiles
#'   \item \code{rhc.gpz}: numeric vector of random variates
#'   \item \code{hhc.gpz}: numeric vector of hazard values
#' }
#'
#' @references 
#' Sah, L.B., & Kumar, V. (2019). 
#' Half-Cauchy Gompertz Distribution : Different Methods of Estimation, 
#' \emph{Journal of  National Academy of Mathematics}, \bold{33},  51--65.

#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dhc.gpz(x, 2.0, 0.5, 2.5)
#' phc.gpz(x, 2.0, 0.5, 2.5)
#' qhc.gpz(0.5, 2.0, 0.5, 2.5)
#' rhc.gpz(10, 2.0, 0.5, 2.5)
#' hhc.gpz(x, 2.0, 0.5, 2.5)
#' 
#' # Data
#' x <- stress66
#' # ML estimates
#' params = list(alpha=1.6660, lambda=0.0328, theta=2.0578)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.gpz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.gpz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params, dfun=dhc.gpz, pfun=phc.gpz, plot=TRUE)
#' print.gofic(out)
#' 
#' @name HCGompertz
#' @aliases dhc.gpz phc.gpz qhc.gpz rhc.gpz hhc.gpz
#' @export
NULL

#' @rdname HCGompertz
#' @usage NULL
#' @export   
dhc.gpz <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- alpha * x[valid]		
	z <- (lambda/alpha) * (1.0 - exp(u))	
    pdf[valid] <- (2/pi)*(lambda/theta)*exp(u) * (1/(1+(-z/theta)^2))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCGompertz
#' @usage NULL
#' @export
phc.gpz <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -(lambda/(theta*alpha))*(1- exp(alpha * q[valid]))	
    cdf[valid] <- (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCGompertz
#' @usage NULL
#' @export
qhc.gpz <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <-  (1/alpha) * log(1.0+ ((alpha*theta)/lambda)*tan(p[valid] * pi *0.5)) 	   
    }    
    return(qtl)  
 } 

#' @rdname HCGompertz
#' @usage NULL
#' @export
rhc.gpz <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.gpz(u, alpha, lambda, theta)
    return(q)
}

#' @rdname HCGompertz
#' @usage NULL
#' @export
hhc.gpz <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dhc.gpz(x, alpha, lambda, theta) 
	surv <- 1.0 - phc.gpz(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 


