# LogisNHE.R
#' Logistic-NHE Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-NHE distribution.
#'
#' The Logistic-NHE distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-NHE distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{1+\left[\exp \left\{(1+\lambda x)^\beta-1\right\}-1\right]^\alpha}  ; x \geq 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' Included functions are:
#' \itemize{
#'   \item \code{dlogis.NHE()} — Density function
#'   \item \code{plogis.NHE()} — Distribution function
#'   \item \code{qlogis.NHE()} — Quantile function
#'   \item \code{rlogis.NHE()} — Random generation
#'   \item \code{hlogis.NHE()} — Hazard function
#' }
#'
#' @usage
#' dlogis.NHE(x, alpha, beta, lambda, log = FALSE)
#' plogis.NHE(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.NHE(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.NHE(n, alpha, beta, lambda)
#' hlogis.NHE(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.NHE}: numeric vector of (log-)densities
#'   \item \code{plogis.NHE}: numeric vector of probabilities
#'   \item \code{qlogis.NHE}: numeric vector of quantiles
#'   \item \code{rlogis.NHE}: numeric vector of random variates
#'   \item \code{hlogis.NHE}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary,A.K., & Kumar, V.(2020). 
#' The Logistic NHE Distribution with Properties and Applications. 
#' \emph{International Journal for Research in Applied Science & Engineering 
#' Technology (IJRASET)}, \bold{8(12)},591--603. 
#' \doi{10.22214/ijraset.2020.32565}   
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dlogis.NHE(x, 2.0, 5.0, 0.2)
#' plogis.NHE(x, 2.0, 5.0, 0.1)
#' qlogis.NHE(0.5, 2.0, 5.0, 0.1)
#' rlogis.NHE(10, 2.0, 5.0, 0.1)
#' hlogis.NHE(x, 2.0, 5.0, 0.1)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(alpha=4.39078, beta=6.98955, lambda=0.01133)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.NHE, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.NHE, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.NHE, pfun=plogis.NHE, plot=TRUE)
#' print.gofic(out) 
#' 
#' @name LogisNHE
#' @aliases dlogis.NHE plogis.NHE qlogis.NHE rlogis.NHE hlogis.NHE
#' @export
NULL

#' @rdname LogisNHE
#' @usage NULL
#' @export   
dlogis.NHE <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- (1+lambda*x[valid])
	w <- exp(u^beta-1)	
    num <- alpha * beta * lambda *(u^(beta-1)) * w * ((w-1)^(alpha-1))
    deno <- (1.0 + (w - 1.0) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LogisNHE
#' @usage NULL
#' @export     
plogis.NHE <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp((1+lambda * q[valid])^ beta -1) 
	tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0 - (1.0/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisNHE
#' @usage NULL
#' @export   
qlogis.NHE <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- (1/lambda) * (((log(1 + ((p[valid]/(1-p[valid])) ^ (1/alpha)))+1)^ (1/beta)) - 1)      
    }    
    return(qtl)   
 } 

#' @rdname LogisNHE
#' @usage NULL
#' @export   
rlogis.NHE <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.NHE(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisNHE
#' @usage NULL
#' @export   
hlogis.NHE <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
    
    nume <- dlogis.NHE(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.NHE(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 

