# ModGE.R
#' Modified Generalized Exponential (MGE) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Modified Generalized Exponential distribution.
#'
#' The Modified Generalized Exponential distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Modified Generalized Exponential distribution has CDF:
#'
#' \deqn{
#' F(x;\alpha,\beta,\lambda)=\left[1-\exp\left\{1-\left(\exp(\beta x)\right)
#' ^{\alpha}\right\}\right]^{\lambda}, \quad x>0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dmod.gen.exp()} — Density function
#'   \item \code{pmod.gen.exp()} — Distribution function
#'   \item \code{qmod.gen.exp()} — Quantile function
#'   \item \code{rmod.gen.exp()} — Random generation
#'   \item \code{hmod.gen.exp()} — Hazard function
#' }
#'
#' @usage
#' dmod.gen.exp(x, alpha, beta, lambda, log = FALSE)
#' pmod.gen.exp(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qmod.gen.exp(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rmod.gen.exp(n, alpha, beta, lambda)
#' hmod.gen.exp(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dmod.gen.exp}: numeric vector of (log-)densities
#'   \item \code{pmod.gen.exp}: numeric vector of probabilities
#'   \item \code{qmod.gen.exp}: numeric vector of quantiles
#'   \item \code{rmod.gen.exp}: numeric vector of random variates
#'   \item \code{hmod.gen.exp}: numeric vector of hazard values
#' }
#'
#' @references
#' Telee, L. B. S., & Kumar, V. (2023). 
#' Modified Generalized Exponential Distribution.  
#' \emph{Nepal Journal ofMathematical Sciences}, \bold{4(1)}, 21–32. 
#' \doi{10.3126/njmathsci.v4i1.53154}
#'
#' Chaudhary, A. K., Sapkota, L. P., & Kumar, V.(2021). 
#' Some Properties and Application of Arctan Generalized Exponential Distribution. 
#' \emph{International Journal of Innovative Research in Science, 
#' Engineering and Technology (IJIRSET)}, \bold{10(1)},456--468.
#'
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dmod.gen.exp(x, 2.0, 0.5, 0.2)
#' pmod.gen.exp(x, 2.0, 0.5, 0.2)
#' qmod.gen.exp(0.5, 2.0, 0.5, 0.2)
#' rmod.gen.exp(10, 2.0, 0.5, 0.2)
#' hmod.gen.exp(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- stress  
#' # ML estimates
#' params = list(alpha=3.1502, beta=0.2167, lambda=0.3636)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pmod.gen.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qmod.gen.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dmod.gen.exp, pfun=pmod.gen.exp, plot=FALSE)
#' print.gofic(out)
#' 
#' @name ModGE
#' @aliases dmod.gen.exp pmod.gen.exp qmod.gen.exp rmod.gen.exp hmod.gen.exp
#' @export
NULL

#' @rdname ModGE
#' @usage NULL
#' @export  
dmod.gen.exp <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- exp(beta * x[valid])	
	z <- exp(-lambda * x[valid] * u)	
    pdf[valid] <- alpha * lambda * (1 + (beta * x[valid])) * u * z * ((1-z) ^ (alpha-1))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname ModGE
#' @usage NULL
#' @export 
pmod.gen.exp <- function (q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- lambda * q[valid] * exp(beta * q[valid])
    cdf[valid] <- (1.0 - exp(-u)) ^ alpha  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname ModGE
#' @usage NULL
#' @export 
qmod.gen.exp <- function (p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	lambda * x * exp(beta * x) + log(1 - wi ^ (1/alpha))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname ModGE
#' @usage NULL
#' @export  
rmod.gen.exp <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qmod.gen.exp(u, alpha, beta, lambda)
    return(q)
}

#' @rdname ModGE
#' @usage NULL
#' @export 
hmod.gen.exp <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dmod.gen.exp(x, alpha, beta, lambda) 
	surv <- 1.0 - pmod.gen.exp(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}
 
