# ModInvLomax.R
#' Modified Inverse Lomax (MIL) Distribution
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Modified Inverse Lomax distribution.
#'
#' The distribution is parameterized by shape parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0} and scale/shape parameter
#' \eqn{\lambda > 0}.
#'
#' @details
#' The cumulative distribution function (CDF) of the MIL distribution is
#'
#' \deqn{
#' F(x; \alpha,\beta,\lambda) =
#' \left[1+\left(\frac{\beta}{x}\right)e^{-\lambda x}\right]^{-\alpha},
#' \quad x>0.
#' }
#'
#' @usage
#' dmod.inv.lomax(x, alpha, beta, lambda, log = FALSE)
#' pmod.inv.lomax(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qmod.inv.lomax(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rmod.inv.lomax(n, alpha, beta, lambda)
#' hmod.inv.lomax(x, alpha, beta, lambda)
#'
#' @param x numeric vector of strictly positive quantiles.
#' @param q numeric vector of strictly positive quantiles.
#' @param p numeric vector of probabilities with values in (0, 1).
#' @param n number of observations (positive integer).
#' @param alpha positive shape parameter.
#' @param beta positive scale parameter.
#' @param lambda positive shape/scale parameter.
#' @param log logical; if \code{TRUE}, returns the log-density.
#' @param lower.tail logical; if \code{TRUE} (default), probabilities are
#'   \eqn{P(X \le x)}, otherwise \eqn{P(X > x)}.
#' @param log.p logical; if \code{TRUE}, probabilities are returned as \code{log(p)}.
#'
#' @return
#' \itemize{
#'   \item \code{dmod.inv.lomax}: numeric vector of (log) densities.
#'   \item \code{pmod.inv.lomax}: numeric vector of distribution function values.
#'   \item \code{qmod.inv.lomax}: numeric vector of quantiles.
#'   \item \code{rmod.inv.lomax}: numeric vector of random variates.
#'   \item \code{hmod.inv.lomax}: numeric vector of hazard rates.
#' }
#'
#' @references
#' Telee, L.B.S., Yadav, R.S., & Kumar V.(2023). 
#' Modified Inverse Lomax Distribution: Model and properties. 
#' \emph{Discovery}, \bold{59: e110d1352}. 
#' \doi{10.54905/disssi.v59i333.e110d1352}
#'
#' @examples
#' x <- seq(0.1, 5, by = 0.1)
#' dmod.inv.lomax(x, alpha = 1.5, beta = 2, lambda = 0.5)
#' pmod.inv.lomax(x, alpha = 1.5, beta = 2, lambda = 0.5)
#' qmod.inv.lomax(0.5, alpha = 1.5, beta = 2, lambda = 0.5)
#' set.seed(123)
#' rmod.inv.lomax(5, alpha = 1.5, beta = 2, lambda = 0.5)
#' hmod.inv.lomax(x, alpha = 1.5, beta = 2, lambda = 0.5)
#'
#' # Data
#' x <- windshield
#' # ML estimates
#' params = list(alpha=0.6661, beta=26.8875, lambda=1.0004)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pmod.inv.lomax, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qmod.inv.lomax, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dmod.inv.lomax, pfun=pmod.inv.lomax, plot=FALSE)
#' print.gofic(out)
#' 
#' @name ModInvLomax
#' @aliases dmod.inv.lomax pmod.inv.lomax qmod.inv.lomax rmod.inv.lomax hmod.inv.lomax
#' @export
NULL

#' @rdname ModInvLomax
#' @usage NULL
#' @export
dmod.inv.lomax <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- lambda * x[valid]	
	w <- exp(-u)
	z <- 1 + (beta / x[valid]) * w
    pdf[valid] <- alpha * (beta/(x[valid] * x[valid])) * (1 + u) * w * (z ^ -(alpha + 1))
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname ModInvLomax
#' @usage NULL
#' @export
pmod.inv.lomax <- function (q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	z <- 1 + (beta/q[valid]) * exp(-lambda * q[valid])
    cdf[valid] <- (z ^ -alpha)   
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname ModInvLomax
#' @usage NULL
#' @export
qmod.inv.lomax <- function (p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	lambda * x + log(x)-log(beta) + log((wi ^ (-1/alpha))-1)   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10 
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 
 
#' @rdname ModInvLomax
#' @usage NULL
#' @export
rmod.inv.lomax <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qmod.inv.lomax(u, alpha, beta, lambda)
    return(q)
}

#' @rdname ModInvLomax
#' @usage NULL
#' @export
hmod.inv.lomax <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dmod.inv.lomax(x, alpha, beta, lambda)
	surv <- 1 - pmod.inv.lomax(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
