# PoissonChen.R
#' Poisson-Chen Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson-Chen distribution.
#'
#' The Poisson-Chen distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson-Chen distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1 - \frac{1}{{1 - {e^{ - \lambda }}}}\left[ {1 - \exp \left\{ 
#' { - \lambda \,\,{e^{\beta (1 - {e^{{x^\alpha }}})\,}}} \right\}} 
#' \right]\quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dpois.chen()} — Density function
#'   \item \code{ppois.chen()} — Distribution function
#'   \item \code{qpois.chen()} — Quantile function
#'   \item \code{rpois.chen()} — Random generation
#'   \item \code{hpois.chen()} — Hazard function
#' }
#'
#' @usage
#' dpois.chen(x, alpha, beta, lambda, log = FALSE)
#' ppois.chen(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.chen(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.chen(n, alpha, beta, lambda)
#' hpois.chen(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.chen}: numeric vector of (log-)densities
#'   \item \code{ppois.chen}: numeric vector of probabilities
#'   \item \code{qpois.chen}: numeric vector of quantiles
#'   \item \code{rpois.chen}: numeric vector of random variates
#'   \item \code{hpois.chen}: numeric vector of hazard values
#' }
#'
#' @references
#' Joshi, R. K., & Kumar, V. (2021).  
#' Poisson Chen Distribution: Properties and Application. 
#' \emph{ International Journal of Latest Trends in Engineering and Technology}, 
#' \bold{18(4)}, 1--12.  
#'
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dpois.chen(x, 2.0, 0.5, 0.2)
#' ppois.chen(x, 2.0, 0.5, 0.2)
#' qpois.chen(0.5, 2.0, 0.5, 0.2)
#' rpois.chen(10, 2.0, 0.5, 0.2)
#' hpois.chen(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- fibers63
#' # ML estimates
#' params = list(alpha=0.53679, beta=1.00238, lambda=108.22948)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.chen, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.chen, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.chen, pfun=ppois.chen, plot=TRUE)
#' print.gofic(out)
#' 
#' @name PoissonChen
#' @aliases dpois.chen ppois.chen qpois.chen rpois.chen hpois.chen
#' @export
NULL

#' @rdname PoissonChen
#' @usage NULL
#' @export 

dpois.chen <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	z <- exp(beta*(1- exp(x[valid] ^ alpha)))
    gx <- alpha * beta * lambda *(x[valid] ^(alpha-1)) * (exp(x[valid]^alpha)) * z * exp(-lambda*z)	
    pdf[valid] <- tmp * gx 
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonChen
#' @usage NULL
#' @export 
ppois.chen <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	z <- exp(beta*(1- exp(q[valid] ^ alpha)))
    cdf[valid] <- 1.0 - tmp * (1-exp(-lambda * z))
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonChen
#' @usage NULL
#' @export
qpois.chen <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	z <- 1+(1/lambda) * log(1-(1-exp(-lambda))*(1-p[valid]))		
    qtl[valid] <- (log(1-(1/beta) * log(1-z))) ^ (1/alpha)    
    }    
    return(qtl)   
 } 

#' @rdname PoissonChen
#' @usage NULL
#' @export 
rpois.chen <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1 || n <= 0 || n != as.integer(n))
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', 'beta' and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, beta and lambda must be positive.")

    u <- runif(n)
    q <- qpois.chen(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonChen
#' @usage NULL
#' @export
hpois.chen <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")

	nume <- dpois.chen(x, alpha, beta, lambda)
	deno <- 1.0 - ppois.chen(x, alpha, beta, lambda)
	hrf <- nume/deno
    return(hrf)    
} 
