# PoissonSGZ.R
#' Poisson Shifted Gompertz (PSG) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson Shifted Gompertz distribution.
#'
#' The Poisson Shifted Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson Shifted Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \frac{1}{\left(1-e^{-\lambda}\right)}\left\{1-\exp 
#' \left[-\lambda\left\{1-\left(1-e^{-\beta x}\right) 
#' \exp \left(-\alpha e^{-\beta x}\right)\right\}\right]\right\}\quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dpois.shifted.gz()} — Density function
#'   \item \code{ppois.shifted.gz()} — Distribution function
#'   \item \code{qpois.shifted.gz()} — Quantile function
#'   \item \code{rpois.shifted.gz()} — Random generation
#'   \item \code{hpois.shifted.gz()} — Hazard function
#' }
#'
#' @usage
#' dpois.shifted.gz(x, alpha, beta, lambda, log = FALSE)
#' ppois.shifted.gz(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.shifted.gz(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.shifted.gz(n, alpha, beta, lambda)
#' hpois.shifted.gz(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.shifted.gz}: numeric vector of (log-)densities
#'   \item \code{ppois.shifted.gz}: numeric vector of probabilities
#'   \item \code{qpois.shifted.gz}: numeric vector of quantiles
#'   \item \code{rpois.shifted.gz}: numeric vector of random variates
#'   \item \code{hpois.shifted.gz}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary,A.K., & Kumar, V. (2021). 
#' Poisson Shifted Gompertz Distribution: Properties and Applications. 
#' \emph{ International Journal of Recent Technology and Engineering (IJRTE)}
#' ,\bold{9(5)},202--208. \doi{10.35940/ijrte.E5265.019521} 
#' 
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dpois.shifted.gz(x, 2.0, 0.5, 0.2)
#' ppois.shifted.gz(x, 2.0, 0.5, 0.2)
#' qpois.shifted.gz(0.5, 2.0, 0.5, 0.2)
#' rpois.shifted.gz(10, 2.0, 0.5, 0.2)
#' hpois.shifted.gz(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- fibers63  
#' # ML estimates
#' params = list(alpha=13.5877, beta=2.0139, lambda=18.8875)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.shifted.gz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.shifted.gz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.shifted.gz, pfun=ppois.shifted.gz, plot=FALSE)
#' print.gofic(out)
#' 
#' @name PoissonSGZ
#' @aliases dpois.shifted.gz ppois.shifted.gz qpois.shifted.gz rpois.shifted.gz hpois.shifted.gz
#' @export
NULL

#' @rdname PoissonSGZ
#' @usage NULL
#' @export  
dpois.shifted.gz <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	u <- exp(-beta*x[valid])
	w <- exp(-alpha*u)
	z <- (1-u)* w
    gx <- beta * lambda * u * w * exp(-lambda*(1-z)) * (1+alpha*(1-u))	
    pdf[valid] <- tmp * gx
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonSGZ
#' @usage NULL
#' @export  
ppois.shifted.gz <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
	tmp <- 1/(1 - exp(-lambda))
	z <- (1-exp(-beta*q[valid]))* exp(-alpha* exp(-beta*q[valid]))
	cdf[valid] <- 1.0 - tmp * (1-exp(-lambda * (1-z)))
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonSGZ
#' @usage NULL
#' @export
qpois.shifted.gz <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	log(1 - exp(-beta * x)) - alpha * exp(-beta * x) - log(1+(1/lambda) * log(1-(1-exp(-lambda))*(1-wi)))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }
  return(qtl)
} 
#' @rdname PoissonSGZ
#' @usage NULL
#' @export
rpois.shifted.gz <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.shifted.gz(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonSGZ
#' @usage NULL
#' @export
## Hazard function of Poisson Shifted Gompertz distribution
hpois.shifted.gz <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dpois.shifted.gz(x, alpha, beta, lambda)
	surv <- 1.0 - ppois.shifted.gz(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
