#' Generic Goodness-of-Fit(GoF) and Model Diagnostics Function
#'
#' Computes log-likelihood, information criteria (AIC, BIC, AICC, HQIC)
#' and classical goodness-of-fit statistics (Kolmogorov–Smirnov,
#' Cramér–von Mises, Anderson–Darling) for a given numeric data vector
#' and user-supplied density and distribution functions.
#'
#' Optionally plots the empirical cumulative distribution function (ECDF)
#' against the theoretical cumulative distribution function.
#'
#' @param x Numeric vector of observed data. Must contain at least two values.
#' @param params Named list of model parameters passed to \code{dfun} and \code{pfun}.
#' @param dfun A probability density function with signature
#'   \code{dfun(x, ...)} returning numeric densities.
#' @param pfun A cumulative distribution function with signature
#'   \code{pfun(q, ...)} returning cumulative probabilities.
#' @param plot Logical; if \code{TRUE}, plots empirical vs theoretical CDF.
#'   Default is \code{TRUE}.
#' @param verbose Logical; if \code{TRUE}, prints the output object.
#'   Default is \code{FALSE}.
#'
#' @return
#' An object of class \code{"gofic"} containing:
#' \itemize{
#'   \item \code{logLik} Numeric; log-likelihood value.
#'   \item \code{AIC} Akaike Information Criterion.
#'   \item \code{BIC} Bayesian Information Criterion.
#'   \item \code{AICC} Corrected Akaike Information Criterion.
#'   \item \code{HQIC} Hannan–Quinn Information Criterion.
#'   \item \code{KS} Object returned by \code{stats::ks.test()}.
#'   \item \code{CVM} Object returned by \code{goftest::cvm.test()}.
#'   \item \code{AD} Object returned by \code{goftest::ad.test()}.
#'   \item \code{n} Sample size.
#'   \item \code{params} Model parameters supplied.
#' }
#'
#' The object is returned invisibly.
#'
#' @details
#' The supplied \code{dfun} and \code{pfun} must accept arguments
#' \code{x} and \code{q} respectively, followed by named model parameters.
#' Density values must be finite and positive; non-positive densities
#' trigger a warning but computation proceeds.
#'
#' @seealso
#' \code{\link{print.gofic}},
#' \code{\link[stats]{ks.test}},
#' \code{\link[goftest]{cvm.test}},
#' \code{\link[goftest]{ad.test}}
#'
#' @examples
#' # Example 1 with built-in Weibull distribution
#' \donttest{
#' set.seed(123)
#' x <- rweibull(100, shape = 2, scale = 1)
#' out <- gofic(x, params = list(shape = 2, scale = 1),
#'       dfun = dweibull, pfun = pweibull, plot=FALSE)
#' out
#' }
#'
#' # Example 2: For a user defined distribution
#' # Goodness-of-Fit(GoF) and Model Diagnostics for Chen-Exponential distribution
#' #Data
#' x <- stress
#' #ML Estimates    
#' params = list(alpha=2.5462, beta=0.0537, lambda=87.6028)
#' # Display plot and print numerical summary
#' gofic(x, params = params,
#'         dfun = dchen.exp, pfun = pchen.exp, plot = TRUE, verbose = TRUE)
#' 
#' # Display plot only (no numerical summary)
#' gofic(x, params = params,
#'         dfun = dchen.exp, pfun = pchen.exp, plot = TRUE, verbose = FALSE)
#' 
#' # Print numerical summary only (no plot)
#' gofic(x, params = params,
#'         dfun = dchen.exp, pfun = pchen.exp, plot = FALSE, verbose = TRUE)
#' 
#' # Display plot; numerical summary stored in 'out'
#' out <- gofic(x, params = params,
#'               dfun = dchen.exp, pfun = pchen.exp, plot = TRUE, verbose = FALSE)
#' print.gofic(out)
#' 
#' # Neither plot nor console output; results stored in 'out'
#' out <- gofic(x, params = params,
#'                dfun = dchen.exp, pfun = pchen.exp, plot = FALSE, verbose = FALSE)
#' print.gofic(out)
#'
#' @export

gofic <- function(x, params, dfun, pfun, plot = TRUE, verbose = FALSE) {
  stopifnot(is.numeric(x), length(x) > 1)
  if (!is.list(params) || is.null(names(params)))
    stop("'params' must be a named list, e.g. list(alpha=1, lambda=2).",
         call. = FALSE)

  if (!is.function(dfun) || !is.function(pfun))
    stop("'dfun' and 'pfun' must be valid R functions.", call. = FALSE)

  n <- length(x)
  p <- length(params)

  args_pdf <- c(list(x = x), params)
  f <- do.call(dfun, args_pdf)

  if (any(f <= 0 | !is.finite(f)))
    warning("Some densities are non-positive or invalid.", call. = FALSE)

  logLik <- sum(log(f[f > 0]))
  AIC  <- -2 * logLik + 2 * p
  BIC  <- -2 * logLik + p * log(n)
  AICC <- AIC + (2 * p * (p + 1)) / (n - p - 1)
  HQIC <- -2 * logLik + 2 * p * log(log(n))

  suppressWarnings({
    KS  <- ks.test(x, function(q) do.call(pfun, c(list(q = q), params)))
  })
  CVM <- goftest::cvm.test(x, function(q) do.call(pfun, c(list(q = q), params)))
  AD  <- goftest::ad.test(x, function(q) do.call(pfun, c(list(q = q), params)))

  if (isTRUE(plot)) {
    xx <- seq(min(x), max(x), length.out = 200)

    plot.ecdf(x,
      main = "Empirical vs Theoretical CDF",
      xlab = "x", ylab = "Cumulative Probability",
      col = "blue", lwd = 2, las = 1
    )

    lines(xx, do.call(pfun, c(list(q = xx), params)),
          col = "red", lwd = 2)

    legend("bottomright",
           legend = c("Empirical", "Theoretical"),
           col = c("blue", "red"), lwd = 2, bty = "n")

    text_label <- sprintf("KS = %.4f\np = %.4f", KS$statistic, KS$p.value)
    text(quantile(x, 0.65), 0.15, labels = text_label,
         adj = 0, cex = 0.9, col = "darkgreen", font = 2)
  }

  out <- list(
    logLik = logLik,
    AIC = AIC, BIC = BIC, AICC = AICC, HQIC = HQIC,
    KS = KS, CVM = CVM, AD = AD,
    n = n, params = params
  )
  class(out) <- "gofic"

  if (isTRUE(verbose)) print(out)

  invisible(out)
}
