#' Fisher information matrix for a two-component Gaussian mixture (working approximation).
#'
#' Computes a Fisher information matrix approximation based on the outer product
#' of gradients for a two-component univariate Gaussian mixture model.
#'
#' @param data Numeric vector of observations.
#' @param params List with elements \code{pi}, \code{mu1}, \code{mu2}, \code{sigma1}, \code{sigma2}.
#'
#' @return Fisher information approximation
#'
#' @keywords internal
#' @importFrom stats dnorm
FIM_mixture_normals <- function(data, params) {
  pi <- params$pi
  mu1 <- params$mu1
  mu2 <- params$mu2
  sigma1 <- params$sigma1
  sigma2 <- params$sigma2

  n <- length(data)
  f1 <- dnorm(data, mean = mu1, sd = sigma1)
  f2 <- dnorm(data, mean = mu2, sd = sigma2)
  w <- (pi * f1) / (pi * f1 + (1 - pi) * f2)
  grad_mu1 <- w * (data - mu1) / sigma1^2
  grad_mu2 <- (1 - w) * (data - mu2) / sigma2^2
  grad_sigma1 <- w * ((data - mu1)^2 - sigma1^2) / sigma1^3
  grad_sigma2 <- (1 - w) * ((data - mu2)^2 - sigma2^2) / sigma2^3
  grad_pi <- (w - pi) / (pi * (1 - pi))

  grad <- cbind(grad_mu1, grad_mu2, grad_sigma1, grad_sigma2, grad_pi)

  fisher_info <- t(grad)%*%grad
  colnames(fisher_info) <- rownames(fisher_info) <- c("mu1", "mu2", "sigma1", "sigma2", "pi")
  return(fisher_info)
}
