# =============================================================================
# Source file: function definitions for PTL simulation
# =============================================================================

#' Outcome nuisance function (linear, target model)
#'
#' @param X Design matrix; dimension n x p.
#' @param Beta Coefficient vector (column matrix); dimension p x 1.
#' @return Vector of outcome nuisance values; dimension n x 1.
f_0 <- function(X, Beta) {
  X %*% Beta
}

#' Treatment nuisance / confounding function (linear, target model)
#'
#' @param X Design matrix; dimension n x p.
#' @param Gamma Coefficient vector (column matrix); dimension p x 1.
#' @return Vector of treatment equation values; dimension n x 1.
g_0 <- function(X, Gamma) {
  X %*% Gamma
}

#' Outcome nuisance function (linear, source model)
#'
#' @param X Design matrix; dimension n x p.
#' @param Beta Coefficient vector (column matrix); dimension p x 1.
#' @return Vector of outcome nuisance values; dimension n x 1.
f_k <- function(X, Beta) {
  X %*% Beta
}

#' Treatment nuisance / confounding function (linear, source model)
#'
#' @param X Design matrix; dimension n x p.
#' @param Gamma Coefficient vector (column matrix); dimension p x 1.
#' @return Vector of treatment equation values; dimension n x 1.
g_k <- function(X, Gamma) {
  X %*% Gamma
}

#' Data generating process: generate (X, D, Y)
#'
#' @param n Sample size.
#' @param q Causal dimension (unused in current body; kept for interface).
#' @param p Nuisance dimension (number of covariates).
#' @param p.nonzero Number of non-zero coefficients for Beta and Gamma.
#' @param rho Causal effect (scalar).
#' @param Beta Nuisance coefficient vector for outcome; dimension p x 1.
#' @param Gamma Nuisance coefficient vector for treatment (confounding); dimension p x 1.
#' @param mu Mean vector of X; length p.
#' @param sigma Base for AR(1)-type covariance: Sigma[i,j] = sigma^|i-j|.
#' @param f_func Function f(X, Beta) for outcome nuisance; default f_0.
#' @param g_func Function g(X, Gamma) for treatment equation; default g_0.
#' @param seed Optional random seed.
#' @return List with components: X (design matrix), D (treatment), Y (outcome), data = cbind(Y, D, X).
DGP <- function(n, q, p, p.nonzero,
                rho, Beta, Gamma,
                mu = rep(10, p), sigma = 0.5,
                f_func = f_0, g_func = g_0,
                seed = NULL) {
  if (!is.null(seed)) set.seed(seed)
  Sig <- matrix(0, p, p)
  for (i in 1:p) {
    for (j in 1:p) {
      Sig[i, j] <- sigma^(abs(i - j))
    }
  }
  X <- MASS::mvrnorm(n, mu, Sig)
  V <- matrix(stats::rnorm(n))
  D <- g_func(X[,1:p.nonzero], Gamma[1:p.nonzero]) + V
  U <- matrix(stats::rnorm(n))
  Y <- rho * D + f_func(X[,1:p.nonzero], Beta[1:p.nonzero]) + U
  list(D = D, X = X, Y = Y, data = cbind(Y, D, X))
}

#' Root mean squared error of parameter estimates
#'
#' @param Theta True parameter vector; dimension p x 1.
#' @param Theta.hat Matrix of estimates; dimension p x n_sim (n_sim = number of simulations).
#' @return Vector of RMSE per parameter component; length p.
RMSE <- function(Theta, Theta.hat) {
  p <- length(Theta)
  n_sim <- ncol(Theta.hat)
  a <- matrix(0, nrow = p, ncol = n_sim)
  for (i in 1:n_sim) {
    a[, i] <- (Theta.hat[, i] - Theta)^2
  }
  MSE <- rowMeans(a)
  sqrt(MSE)
}
