% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inferP.R
\name{inferP}
\alias{inferP}
\title{Calculate lower and upper probability bounds}
\usage{
inferP(target, ..., solidus = TRUE)
}
\arguments{
\item{target}{The target probability expression (see Details).}

\item{...}{Probability constraints (see Details).}

\item{solidus}{logical. If \code{TRUE} (default), the symbol \code{|} is used to introduce the conditional in the probability; in this case any use of \code{||} for the 'or'-connective will lead to an error. If \code{FALSE}, the symbol \code{~} is used to introduce the conditional; in this case the symbols \code{|}, \code{||} can be used for the 'or`-connective.}
}
\value{
A vector of \code{min} and \code{max} values for the target probability, or \code{NA} if the constraints are mutually contradictory. If \code{min} and \code{max} are \code{0} and \code{1} then the constraints do not restrict the target probability in any way.
}
\description{
\code{inferP()} calculates the minimum and maximum allowed values of the probability for a propositional-logic expression conditional on another one, given numerical or equality constraints for the conditional probabilities for other propositional-logic expressions.
}
\details{
The function takes as first argument the probability for a logical expression, conditional on another expression, and as subsequent (optional) arguments the constraints on the probabilities for other logical expressions. Propositional logic is intended here.

The function uses the \code{\link[lpSolve:lp]{lpSolve::lp()}} function from the \href{https://cran.r-project.org/package=lpSolve}{\strong{lpSolve}} package.
\subsection{Logical expressions}{

A propositional-logic expression is a combination of atomic propositions by means of logical connectives. Atomic propositions can have any name that satisfies \href{https://cran.r-project.org/doc/FAQ/R-FAQ.html#What-are-valid-names_003f}{R syntax for \emph{object names}}. Examples:

\if{html}{\out{<div class="sourceCode">}}\preformatted{a
A
hypothesis1
coin.lands.tails
coin_lands_heads
`tomorrow it rains` # note the backticks
}\if{html}{\out{</div>}}

Available logical connectives are "not" (negation, "\eqn{\lnot}"), "and" (conjunction, "\eqn{\land}"), "or" (disjunction, "\eqn{\lor}"), "if-then" (implication, "\eqn{\Rightarrow}") which internally is simply defined as "x or not-y". Two kinds of notation are available:

\strong{Arithmetic notation}: should be used with argument \code{solidus = TRUE}:
\itemize{
\item Not: \verb{ -}
\item And: \code{*}
\item Or: \code{+}
\item If-then: \code{>}
}

\strong{Logical notation} (see \link[base:logical]{base::logical}): should be used with the argument \code{solidus = FALSE}:
\itemize{
\item Not: \code{!}
\item And: \code{&} or \code{&&}
\item Or: \code{|} or \code{||}
\item If-then: two-argument function \code{ifthen()}
}

The two kinds of notation can actually be mixed (only \code{|} and \code{||} are not allowed if \code{solidus = TRUE}), but \strong{beware} of very unusual connective precedence in that case; for example, \code{!a + b} is interpreted as \code{!(a + b)}. If you use mixed notation you should explicitly group with parentheses to ensure the desired connective precedence. A warning is issued when \code{!} is used for negation together with \code{+}, \code{*}, because the grouping is dangerously counter-intuitive in this case.

Examples of logical expressions:

\if{html}{\out{<div class="sourceCode">}}\preformatted{a
a & b
(a + hypothesis1) & -A
red.ball & ((a > -b) + c)
}\if{html}{\out{</div>}}
}

\subsection{Probabilities of logical expressions}{

The probability of an expression \eqn{X} conditional on an expression \eqn{Y}in entered with syntax similar to the common mathematical notation \eqn{\mathrm{P}(X \vert Y)}{P(X|Y)}. The solidus "\code{|}" is used to separate the conditional (note that in usual R syntax such symbol stands for logical "or" instead). If the argument \code{solidus = FALSE} is given in the function, then the tilde  "\code{~}" is used instead of the solidus (note that in usual R syntax such symbol introduces a formula instead). For instance

\eqn{\mathrm{P}(\lnot a \lor b \:\vert\: c \land H)}{P(not-a or b | c and H)}

can be entered in the following ways, among others (extra spaces added just for clarity):

\if{html}{\out{<div class="sourceCode">}}\preformatted{P(-a + b  |  c * H)
}\if{html}{\out{</div>}}

or, if argument  \code{solidus = FALSE}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{P(!a | b  ~  c & H)
P(!a || b  ~  c && H)
}\if{html}{\out{</div>}}

It is also possible to use \code{Pr}, \code{p}, \code{pr}, \code{F}, \code{f}, \code{T} (for "truth"), or \code{t} instead of \code{P}.
}

\subsection{Probability constraints}{

Each probability constraint can have one of these four forms:

\if{html}{\out{<div class="sourceCode">}}\preformatted{P(X | Z) = [number between 0 and 1]

P(X | Z) = P(Y | Z)

P(X | Z) = P(Y | Z) * [positive number]

P(X | Z) = P(Y | Z) / [positive number]
}\if{html}{\out{</div>}}

where \code{X}, \code{Y}, \code{Z} are logical expressions. Note that the conditionals on the left and right sides must be the same. Inequalities \code{<=} \code{>=} are also allowed instead of equalities.

See the accompanying vignette for more interesting examples.
}
}
\examples{

## No constraints
inferP(
  target = P(a | h)
)

## Trivial example with inequality constraint
inferP(
  target = P(a | h),
  P(-a | h) >= 0.2
)

#' ## The probability of an "and" is always less
## than the probabilities of the and-ed propositions:
inferP(
  target = P(a & b | h),
  P(a | h) == 0.3,
  P(b | h) == 0.6
)

## P(a & b | h) is completely determined
## by P(a | h) and P(b | a & h):
inferP(
    target = P(a & b | h),
    P(a | h) == 0.3,
    P(b | a & h) == 0.2
)

## Logical implication (modus ponens)
inferP(
  target = P(b | I),
  P(a | I) == 1,
  P(a > b | I) == 1
)

## Cut rule of sequent calculus
inferP(
  target = P(X + Y | I & J),
  P(A & X | I) == 1,
  P(Y | A & J) == 1
)

## Solution to the Monty Hall problem (see accompanying vignette):
inferP(
    target = P(car2  |  you1 & host3 & I),
    ##
    P(car1 & car2  |  I) == 0,
    P(car1 & car3  |  I) == 0,
    P(car2 & car3  |  I) == 0,
    P(car1 + car2 + car3  |  I) == 1,
    P(host1 & host2 | I) == 0,
    P(host1 & host3 | I) == 0,
    P(host2 & host3 | I) == 0,
    P(host1 + host2 + host3  |  I) == 1,
    P(host1  |  you1 & I) == 0,
    P(host2  |  car2 & I) == 0,
    P(host3  |  car3 & I) == 0,
    P(car1  |  I) == P(car2  |  I),
    P(car2  |  I) == P(car3  |  I),
    P(car1  |  you1 & I) == P(car2  |  you1 & I),
    P(car2  |  you1 & I) == P(car3  |  you1 & I),
    P(host2  |  you1 & car1 & I) == P(host3  |  you1 & car1 & I)
)

}
\references{
T. Hailperin: \emph{Best Possible Inequalities for the Probability of a Logical Function of Events}. Am. Math. Monthly 72(4):343, 1965 \url{doi:10.1080/00029890.1965.11970533}.

T. Hailperin: \emph{Sentential Probability Logic: Origins, Development, Current Status, and Technical Applications}. Associated University Presses, 1996 \url{https://archive.org/details/hailperin1996-Sentential_probability_logic/}.
}
