#' Check Technical Replicates
#'
#' Scans the experiment for wells where technical replicates disagree (high standard deviation).
#'
#' @param data Output from import_plate().
#' @param sd_threshold Maximum allowed Standard Deviation between replicates (default 0.5).
#' @return A dataframe of "Bad Wells" to investigate.
#' @export
#' @examples
#' # Data with a good group (SD=0.1) and a bad group (SD=2.0)
#' df <- data.frame(
#'   Sample = c(rep("S1", 3), rep("S2", 3)),
#'   Gene = "GAPDH",
#'   Ct = c(20.0, 20.1, 20.2,  25.0, 25.0, 29.0) # S2 has an outlier
#' )
#'
#' # Run Check
#' check_replicates(df, sd_threshold = 0.5)
check_replicates <- function(data, sd_threshold = 0.5) {

  qc_summary <- data %>%
    dplyr::group_by(Sample, Gene) %>%
    dplyr::summarise(
      mean_ct = mean(Ct, na.rm = TRUE),
      std_ct = sd(Ct, na.rm = TRUE),
      n_wells = dplyr::n(),
      .groups = "drop"
    )

  bad_groups <- qc_summary %>%
    dplyr::filter(std_ct > sd_threshold) %>%
    dplyr::arrange(dplyr::desc(std_ct))

  if(nrow(bad_groups) > 0) {
    warning(paste("ALERT: Found", nrow(bad_groups), "sample/gene pairs with high variance (> ", sd_threshold, ")"))
    return(bad_groups)
  } else {
    message("QC PASSED: All technical replicates are consistent.")
    return(NULL)
  }
}
