#' Michaelis-Menten equations
#'
#' The standard Michaelis-Menten equations
#'
#' @param t Current time
#' @param x Current value of all nodes
#' @param params List of parameters. Should include the adjacency matrix of the network `M`, and parameters `f, h,` and `B`
#'
#' @return A list with `dx`
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'  
#'    Rato::node.removal.thread(  Rato::Michaelis.Menten
#'                              , g$M
#'                              , g$initial_values
#'                              , initial_params = list('f' = 1, 'h'=2, 'B'=0.1))
#'
#' }
Michaelis.Menten <- function(t, x, params)
{
  A <- params$M
  f <- params$f
  h <- params$h
  B <- params$B

  dx <- -1 * B*x^f +  A %*% ((x^h)/(x^h + 1))
  return(list(dx))
}

#' Optimized Michaelis-Menten equations
#'
#' The optimized Michaelis-Menten equations described by Barabasi in his paper 'Universal resilience patterns in complex networks'.
#'
#' @param t Current time
#' @param x Current value of all nodes
#' @param params List of parameters. Should include the B_eff matrix of the network as `Beta`, and parameters `f, h,` and `B`. To understand how to construct the matrix B_eff, please refer to the Barabasi paper.
#'
#' @return A list with `dx`
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'  
#'    initial_parameters <- function(M){
#'      initial_params <- list('f' = 1, 'h' = 2, 'B' = 0.1, 'Beta' = Rato::Barabasi.B_eff(M))
#'      return(initial_params)
#'    }
#'    
#'    update_parameters <- function(params){
#'      M <- params$M
#'      params$Beta <- Rato::Barabasi.B_eff(M)
#'      return(params)
#'    }
#'  
#'    Rato::node.removal.thread( system = Rato::Michaelis.Menten.Fast
#'                             , M = g$M
#'                             , x0 = Rato::Barabasi.X_eff(g$M, g$initial_values)
#'                             , initial_params = initial_parameters
#'                             , update_params = update_parameters
#'                             , to.numeric = TRUE)
#'
#' }
Michaelis.Menten.Fast <- function(t, x, params)
{
  Beta <- params$Beta
  f <- params$f
  h <- params$h
  B <- params$B

  dxdt <- -B * (x^f) +  Beta %*% ((x^h)/(x^h + 1)) # Exemplo de equação: dy/dt = -0.1 * y

  list(dxdt)
}

