#' Distance between groups
#'
#'
#' This function calculates the distance between points in two groups. For each point in the first group, it calculates the distance from that point to all points in the second group.  Finally, it takes the minimum distance obtained.
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param group1 Number. Number of the first group.
#' @param group2 Number. Number of the second group.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param FHW_output List. Output of the \code{Hartigan_and_Wong} function. List with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'
#' @return Returns a number, the value of the minimum distance between pair of points of the two groups.
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'          matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'          matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'
#'FHW_output=Hartigan_and_Wong(data,
#'                             Euclideandistance,
#'                             k,
#'                             centers_function_mean,
#'                             init_centers_random,
#'                             seed=seed,
#'                             10)
#'
#'DistanceBetweenGroups(1, 2, FHW_output, Euclideandistance, data)
#'
#' @export




DistanceBetweenGroups<-function(group1, group2, FHW_output, distance, data){


 G1_points=FHW_output$grouping[[group1]]
 G2_points=FHW_output$grouping[[group2]]

 dist_clust=matrix(0,length(G1_points),length(G2_points) )
 for(i in 1:length(G1_points)){
   for (j in 1:length(G2_points)){
     dist_clust[i,j]=distance(data[G1_points[i],] ,  data[G2_points[j],])
   }
 }
 min_dist_clust=min(dist_clust)

 return(min_dist_clust)
}
