#' Interactive Visualization of Occurrence Flags with mapview
#'
#' @description
#' This function creates an interactive map of occurrence records using
#' **mapview**, visually highlighting flags. This tool helps users explore which
#' records were flagged by one or more validation functions and inspect them
#' directly on the map.
#'
#' @param occ (data.frame or data.table) a dataset containing occurrence records
#' that has been processed by one or more flagging functions. See *Details* for
#' available flag types.
#' @param species (character) name of the species to subset and plot. Default is
#' `NULL`, meaning that all records for all species are plotted.
#' @param long (character) the name of the column in `occ` that contains the
#' longitude values. Default is `"decimalLongitude"`.
#' @param lat (character) the name of the column in `occ` that contains the
#' latitude values. Default is `"decimalLatitude"`.
#' @param flags (character) the flags to be used for coloring the records. Use
#' `"all"` to display all available flags. See *Details* for all options.
#' Default is `"all"`.
#' @param additional_flags (character) an optional named character vector with
#' the names of additional logical columns to be used as flags. Default is `NULL`.
#' @param names_additional_flags (character) an optional different name to the
#' flag provided in `additional_flags` to be shown in the map. Only applicable
#' if `additional_flags` is not NULL.
#' @param col_additional_flags (character) if `additional_flags` is provided,
#' the color of the records flagged in this column. Obligatory if
#' `additional_flags` is not NULL.
#' @param show_no_flagged (logical) whether to display records that did not
#' receive any flag.Default is TRUE.
#' @param cex (numeric) point size for plotting occurrences. Default is 6.
#' @param lwd (numeric) line width for point borders. Default is 2.
#' @param col_points (character) A **named vector** assigning colors to each
#' flag. If `NULL`, default colors from the internal object
#' `RuHere::flag_colors()` are used.
#' @param label (character) column name in `occ` to use as the mouseover label.
#' Default is `NULL`, in which case the record's row number is displayed.
#' @param ... additional arguments passed to `mapview::mapview()`.
#'
#' @details
#' The following flags are available: correct_country, correct_state, cultivated,
#' fossil, inaturalist, faunabr, florabr, wcvp, iucn, duplicated, thin_geo,
#' thin_env, .val, .equ, .zer, .cap, .cen, .sea, .urb, .otl, .gbf, .inst, and
#' .aohi.
#'
#' These flags are typically generated by functions in the `RuHere` or
#' `CoordinateCleaner`workflow to identify potential data-quality issues in
#' occurrence records.
#'
#' Users may also supply additional logical columns using
#' `additional_flags`, optionally providing alternative display names
#' (`names_additional_flags`) and colors (`col_additional_flags`).
#'
#' @returns
#' An interactive mapview object displaying flagged and optionally unflagged
#' occurrence records.
#'
#' @export
#' @importFrom mapview mapview
#'
#' @examples
#' \donttest{
#' # Load example data
#' data("occ_flagged", package = "RuHere")
#' # Visualize flags interactively
#' map_here(occ = occ_flagged, label = "record_id")
#' }
map_here <- function(occ,
                     species = NULL,
                     long = "decimalLongitude",
                     lat = "decimalLatitude",
                     flags = "all",
                     additional_flags = NULL,
                     names_additional_flags = NULL,
                     col_additional_flags = NULL,
                     show_no_flagged = TRUE,
                     cex = 6,
                     lwd = 2,
                     col_points = NULL,
                     label = NULL,
                     ...){

  # ---- Argument checking -----------------------------------------------------

  # occ
  if (!inherits(occ, "data.frame") && !inherits(occ, "data.table")) {
    stop("`occ` must be a data.frame or data.table.", call. = FALSE)
  }

  # Force occ to be a dataframe
  if(inherits(occ, "data.table") | inherits(occ, "spatialvalid"))
    occ <- as.data.frame(occ)

  # flags
  if (!inherits(flags, "character") || length(flags) < 1) {
    stop("`flags` must be a character vector.", call. = FALSE)
  }

  # species
  if(!is.null(species)){
    if (!inherits(species, "character")) {
      stop("`species` must be a character vector.", call. = FALSE)
    }

    if(!species %in% unique(occ$species)){
      stop("The species name provided in 'species' is not present in the 'occ' dataset. It must match a value in the species column.")
    }
    # Subset
    occ <- occ[occ$species %in% species, ]
  }

  # additional_flags must be NULL or a character vector
  if (!is.null(additional_flags) && !is.character(additional_flags)) {
    stop("'additional_flags' must be a character vector.", call. = FALSE)
  }

  # if additional_flags provided, ensure they exist and are logical
  if (!is.null(additional_flags)) {
    # if(is.null(names(additional_flags))){
    #   stop("'additional_flags' must be a named character (i.e., 'user_flag' = 'User flag')")
    # }
    missing_extra <- additional_flags[!additional_flags %in% names(occ)]
    if (length(missing_extra) > 0) {
      stop(
        "These additional flag columns do not exist in 'occ': ",
        paste(missing_extra, collapse = ", "),
        call. = FALSE
      )
    }

    # ensure they are logical
    non_logical <- additional_flags[
      !sapply(occ[, additional_flags, drop = FALSE], is.logical)
    ]
    if (length(non_logical) > 0) {
      stop(
        "These additional flag columns are not logical: ",
        paste(non_logical, collapse = ", "),
        call. = FALSE
      )
    }

    # names_additional_flags must be character
    if (is.null(names_additional_flags) || !inherits(names_additional_flags, "character")) {
      stop("'names_additional_flags' must be a character vector when 'additional_flags' is not NULL.")
    }

    # col_additional_flags must be provided
    if (is.null(col_additional_flags)) {
      stop("'col_additional_flags' must be provided when 'additional_flags' is not NULL.")
    }

    # col_additional_flags must be character
    if (!inherits(col_additional_flags, "character")) {
      stop("'col_additional_flags' must be a character value.")
    }

    # Length consistency checks
    if(!is.null(names_additional_flags)){
      if (length(names_additional_flags) != length(additional_flags)) {
        stop("`names_additional_flags` must have the same length as `additional_flags`.")
    }}

    if (length(col_additional_flags) != length(additional_flags)) {
      stop("`col_additional_flags` must have the same length as `additional_flags`.")
    }

  }


  # show_no_flagged
  if (!inherits(show_no_flagged, "logical") || length(show_no_flagged) != 1) {
    stop("`show_no_flagged` must be a single logical value.", call. = FALSE)
  }

  # cex
  if (!inherits(cex, "numeric") || length(cex) != 1 || cex <= 0) {
    stop("`cex` must be a single positive numeric value.", call. = FALSE)
  }

  # lwd
  if (!inherits(lwd, "numeric") || length(lwd) != 1 || lwd <= 0) {
    stop("`lwd` must be a single positive numeric value.", call. = FALSE)
  }

  # col_points
  if (!is.null(col_points)) {
    if (!inherits(col_points, "character")) {
      stop("`col_points` must be a named character vector of colors.",
           call. = FALSE)
    }
    if (is.null(names(col_points))) {
      stop("`col_points` must have names corresponding to flags.",
           call. = FALSE)
    }
  }

  # label
  if (!is.null(label)) {
    if (!inherits(label, "character") || length(label) != 1) {
      stop("`label` must be a single character string.", call. = FALSE)
    }
    if (!label %in% names(occ)) {
      stop("The column specified in `label` does not exist in `occ`.",
           call. = FALSE)
    }
  }

  if(all(flags == "all")){
    flags <- c("correct_country", "correct_state", "florabr", "faunabr",
               "wcvp", "iucn", "bien", "cultivated", "inaturalist",
               "duplicated", "thin_env", "thin_geo", "consensus",
               "fossil", "year",
               # Froom CoordinateCleaner
               ".val", ".equ", ".zer", ".cap", ".cen", ".sea", ".urb", ".otl",
               ".gbf", ".inst", ".aohi")
  }

  # Add _flags for some columns
  to_paste <- c("florabr", "faunabr", "wcvp", "iucn", "bien", "cultivated",
                "inaturalist", "duplicated", "thin_env", "thin_geo",
                "consensus", "fossil", "year")

  flags[flags %in% to_paste] <- paste0(flags[flags %in% to_paste], "_flag")

  # Additional flags
  if(!is.null(additional_flags)){
    flags <- c(flags, additional_flags)
    }


  # Subset columns
  flags <- intersect(flags, colnames(occ))

  # Remove flags with NAs only
  to_remove <- colSums(is.na(occ[, flags, drop = FALSE])) == nrow(occ)
  to_remove <- names(to_remove)[to_remove]
  occ <- occ[, !(names(occ) %in% to_remove), drop = FALSE]

  #Update flags
  flags <- intersect(flags, colnames(occ))

  # Names of flags
  flag_names <- getExportedValue("RuHere", "flag_names")

  # Names of additional flags
  # Get name
  if(!is.null(additional_flags)){
    if(is.null(names_additional_flags)){
      names_additional_flags <- additional_flags
    }
    names(names_additional_flags) <- additional_flags
    flag_names <- c(flag_names, names_additional_flags)
    }


  # Split points by flag
  occ_list <- lapply(flags, function(i){
    occ[!occ[[i]],]
  })
  #Rename
  names(occ_list) <- flag_names[flags]


  # Remove flags with 0 records
  occ_list <- occ_list[sapply(occ_list, function(x) nrow(x) > 0)]

  # Add non-flagged occurrences?
  if(show_no_flagged){
    no_flagged <- rowSums(occ[, flags, drop = FALSE] |
                            is.na(occ[, flags, drop = FALSE]), na.rm = TRUE)
    no_flagged <- no_flagged == length(flags)
    no_flagged <- occ[no_flagged,]
    occ_list[["No flagged"]] <- no_flagged
  }

  # Spatialize
  occ_list <- lapply(occ_list, spatialize, long = long, lat = lat)

  # Update flags
  flags <- names(occ_list)

  # If colors is NULL, use default color
  if(is.null(col_points)){
    col_points <- getExportedValue("RuHere", "flag_colors")
  }

  # Append color of additional flags
  if(!is.null(additional_flags)){
    col_additional <- col_additional_flags
    names(col_additional) <- names_additional_flags
    col_points <- c(col_points, col_additional)
  }

  # Filter maps with 0 geometries
  occ_list <- occ_list[sapply(occ_list, function(x) length(x) > 0)]

  # Plot first map
  first_map <- names(occ_list)[1]
  mapa <- mapview::mapview(occ_list[[first_map]],
                           col.regions = col_points[[first_map]],
                           cex = cex, lwd = lwd,
                           layer.name = first_map,
                           label = label, ...)
  # Add other maps
  if(length(flags) > 1){
    for(i in 2:length(flags)){
      flag_i <- flags[i]
      mapa <- mapa + mapview::mapview(occ_list[[flag_i]],
                             col.regions = col_points[[flag_i]],
                             cex = cex, lwd = lwd,
                             layer.name = flag_i,
                             label = label, ...)
    }
  }

  return(mapa)
}
