% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/activity.R
\name{activity}
\alias{activity}
\title{Acoustic Activity Matrix}
\usage{
activity(
  soundfile,
  channel = "stereo",
  timeBin = 60,
  dbThreshold = -90,
  targetSampRate = NULL,
  wl = 512,
  window = signal::hamming(wl),
  overlap = ceiling(length(window)/2),
  histbreaks = "FD",
  powthr = 10,
  bgnthr = 0.8,
  beta = TRUE
)
}
\arguments{
\item{soundfile}{tuneR Wave object or path to a valid audio}

\item{channel}{channel where the saturation values will be extract from. Available channels are: \code{"stereo"}, \code{"mono"}, \code{"left"} or \code{"right"}. Defaults to \code{"stereo"}.}

\item{timeBin}{size (in seconds) of the time bin. Set to \code{NULL} to use the entire audio as a single bin. Defaults to \code{60}}

\item{dbThreshold}{minimum allowed value of dB for the spectrograms. Defaults to \code{-90}, as set by Towsey 2017}

\item{targetSampRate}{desired sample rate of the audios.  This argument is only used to down sample the audio. If \code{NULL}, then audio's sample rate remains the same. Defaults to \code{NULL}}

\item{wl}{window length of the spectrogram. Defaults to \code{512}}

\item{window}{window used to smooth the spectrogram. Switch to \code{signal::hanning(wl)} to use hanning instead. Defaults to \code{signal::hammning(wl)}}

\item{overlap}{overlap between the spectrogram windows. Defaults to \code{wl/2} (half the window length)}

\item{histbreaks}{breaks used to calculate Background Noise. Available breaks are: \code{"FD"}, \code{"Sturges"}, \code{"scott"} or any numeric value (foe example = \code{100}). Defaults to \code{"FD"}}

\item{powthr}{single numeric value to calculate the activity matrix for soundscape power (in dB). Detauls to \code{10}}

\item{bgnthr}{single numeric value to calculate the activity matrix for background noise (in \%). Detauls to \code{0.8}}

\item{beta}{how BGN thresholds are calculated. If \code{TRUE}, BGN thresholds are calculated using all recordings combined. If FALSE, BGN thresholds are calculated separately for each recording. Defaults to \code{TRUE}}
}
\value{
This function returns a 0 and 1 matrix containing the activity for all time bins of the inputted file. The matrix's number of rows will equal to half the set window lenght (\code{wl}) and number of columns will equal the number of bins. Cells with the value of 1 represent the acoustically active frequency of a bin.
}
\description{
Calculate the Acoustic Activity Matrix using the methodology proposed in Burivalova 2018
}
\details{
To calculate the activity matrix, we use the methodology proposed by Burivalova 2018. We begin by applying the following formula to each time bin of the recording:

\deqn{a_{mf} = 1\  if (BGN_{mf} > \theta_{1})\  or\  (POW_{mf} > \theta_{2});\  otherwise,\  a_{mf} = 0,}

Where \eqn{\theta_{1}} equals the threshold of BGN values and \eqn{\theta_{2}} equals the threshold of dB values. We set 1 to active and 0 to inactive frequency windows.
}
\examples{
if (require("ggplot2")) {
### Generating an artificial audio for the example
## For this example we'll generate a sweep in a noisy soundscape
library(tuneR)
library(ggplot2)

# Define parameters for the artificial audio
samprate <- 12050
dur <- 60
n <- samprate * dur

# White noise
set.seed(413)
noise <- rnorm(n)

# Linear fade-out envelope
fade <- seq(1, 0, length.out = n)

# Apply fade
signal <- noise * fade

# Create Wave object
wave <- Wave(
  left = signal,
  samp.rate = samprate,
  bit = 16
)

# Running singleSat() on the artificial audio
time <- 10
sat <- activity(wave, timeBin = time)

# Now we can plot the results
satDim <- dim(sat)
numericTime <- seq(0, dur, by = time)
labels <- paste0(numericTime[-length(numericTime)], "-", numericTime[-1], "s")

satDF <- data.frame(BIN = rep(paste0("BIN", seq(satDim[2])), each = satDim[1]),
                    WIN = rep(seq(satDim[1]), satDim[2]),
                    ACT = factor(c(sat), levels = c(0,1)))

ggplot(satDF, aes(x = BIN, y = WIN, fill = ACT)) +
  geom_tile() +
  theme_bw() +
  scale_fill_manual(values = c("white", "black")) +
  scale_y_continuous(expand = c(0,0)) +
  scale_x_discrete(expand = c(0,0), labels = labels) +
  labs(x = "Time Bin", y = "Spectral Window") +
  guides(fill = guide_legend(title = "Activity"))

}
}
\references{
Burivalova, Z., Towsey, M., Boucher, T., Truskinger, A., Apelis, C., Roe, P., & Game, E. T. (2018). Using soundscapes to detect variable degrees of human influence on tropical forests in Papua New Guinea. Conservation Biology, 32(1), 205-215. https://doi.org/10.1111/cobi.12968
}
