#' Posterior-Predictive Metrics for Sca-MCMC Fit
#'
#' Computes R2, RMSE, AIC and BIC after discarding the first 50% of every chain
#' as burn-in. Predictive expectations are obtained by plugging the posterior
#' mean coefficients into the appropriate inverse-link function.
#'
#' @param fit Object returned by `sca_mcmc` (must contain `beta_chain` and `family`).
#' @param y Observed response vector (length n).
#' @param X n x p design matrix used for the fit.
#'
#' @return A data.frame with columns: R2, RMSE, AIC, BIC.
#'
#' @examples
#' set.seed(123)
#' X <- matrix(rnorm(100 * 3), 100, 3)
#' beta <- c(0.5, -0.2, 0.1)
#' y <- rgamma(100, shape = 2, rate = exp(X %*% beta))
#' fit <- sca_mcmc(y, X, family = "gamma", method = "ScaII", iter = 1000)
#' vals <- compute_metrics(fit, y, X)
#' print(vals)
#'
#' @export
compute_metrics <- function(fit, y, X) {
  if (!inherits(fit, "SDGLM")) {
    stop("fit must be an object of class SDGLM")
  }
  
  burn <- floor(dim(fit$beta_chain)[1] * 0.5)
  post <- fit$beta_chain[-(1:burn), , , drop = FALSE]
  beta_mean <- apply(post, 2, mean)
  eta <- as.numeric(X %*% beta_mean)
  
  y_pred <- switch(
    fit$family,
    "poisson" = exp(eta),
    "pareto"  = exp(eta) / (exp(eta) - 1),
    "gamma"   = 2 / exp(eta)
  )
  
  R2 <- stats::cor(y, y_pred)^2
  RMSE <- sqrt(mean((y - y_pred)^2))
  logLik <- sum(dglm_likelihood(y, X, beta_mean, fit$family))
  p <- length(beta_mean)
  AIC <- 2 * p - 2 * logLik
  BIC <- -2 * logLik + p * log(length(y))
  
  data.frame(R2 = R2, RMSE = RMSE, AIC = AIC, BIC = BIC)
}
