#' Generate Random Samples from the Inverse Wishart Distribution
#'
#' Implements Bartlett decomposition to sample from the inverse Wishart distribution
#' IW(v, S), where v = degrees of freedom and S = scale matrix.
#'
#' @param n Integer (>=1). Number of inverse Wishart samples to generate (default = 1).
#' @param v Numeric (scalar). Degrees of freedom; must satisfy v > p - 1 (where p = ncol(S)).
#' @param S Numeric matrix (p x p). Positive-definite scale matrix.
#'
#' @return If n=1: p x p inverse Wishart sample matrix.
#'         If n>1: 3D array (p x p x n) of independent inverse Wishart samples.
#'
#' @examples
#' # 1 sample from IW(v=5, S=diag(2))
#' set.seed(123)
#' Sigma <- rinvwishart(n = 1, v = 5, S = diag(2))
#'
#' # 10 samples (3D array)
#' Sigma_arr <- rinvwishart(n = 10, v = 5, S = diag(2))
#'
#' @export
rinvwishart <- function(n = 1, v, S) {
  # Input validation
  n <- as.integer(n)
  if (n < 1) stop("'n' must be >= 1")
  
  if (missing(S) || !is.matrix(S) || !is.numeric(S)) {
    stop("'S' must be a numeric p x p positive-definite matrix")
  }
  p <- ncol(S)
  if (nrow(S) != p) stop("'S' must be a square matrix")
  
  # Symmetrize to avoid numerical issues
  S <- (S + t(S)) / 2
  
  # Check positive definiteness
  eig <- try(eigen(S, only.values = TRUE)$values, silent = TRUE)
  if (inherits(eig, "try-error") || any(eig <= .Machine$double.eps)) {
    stop("'S' must be positive-definite")
  }
  
  if (missing(v) || !is.numeric(v) || length(v) != 1) {
    stop("'v' must be a scalar (degrees of freedom)")
  }
  if (v <= p - 1) {
    stop(sprintf("Degrees of freedom 'v' must be > p-1 (p=%d; v=%g)", p, v))
  }
  
  # Precompute
  Sinv <- solve(S)
  chol_Sinv <- chol(Sinv)
  df_vec <- v - (1:p) + 1
  
  # Initialize output array
  out <- array(NA, dim = c(p, p, n))
  
  # Generate samples
  for (i in 1:n) {
    # Bartlett decomposition
    A <- matrix(0, p, p)
    A[lower.tri(A)] <- stats::rnorm(p * (p - 1) / 2)
    diag(A) <- sqrt(stats::rchisq(p, df = df_vec))
    
    # Compute Wishart matrix W ~ W(v, Sinv)
    W <- chol_Sinv %*% A %*% t(A) %*% t(chol_Sinv)
    
    # Invert to get inverse Wishart sample
    out[, , i] <- solve(W)
  }
  
  # Return format
  if (n == 1) {
    out <- out[, , 1]
  }
  
  return(out)
}
