#' @title Spatial Network Utilities
#'
#' @description
#' Utility functions for building and manipulating spatial neighborhood networks.
#' These functions are used by SVG detection methods to define spatial relationships
#' between spots/cells.
#'
#' @return See individual function documentation for return values.
#' @name utils_spatial
NULL


#' Build Spatial Neighborhood Network
#'
#' @description
#' Constructs a spatial neighborhood network from spatial coordinates using
#' either Delaunay triangulation or K-nearest neighbors (KNN) approach.
#'
#' @param coords Numeric matrix of spatial coordinates.
#'   Rows are spatial locations, columns are coordinate dimensions (typically x, y).
#' @param method Character string specifying the network construction method.
#'   \itemize{
#'     \item \code{"delaunay"}: Delaunay triangulation (default). Creates a network
#'       where neighbors are determined by triangulation. Works well for relatively

#'       uniform spatial distributions.
#'     \item \code{"knn"}: K-nearest neighbors. Each spot is connected to its k
#'       nearest neighbors based on Euclidean distance.
#'   }
#' @param k Integer. Number of nearest neighbors for KNN method. Default is 10.
#'   Ignored when \code{method = "delaunay"}.
#' @param filter_dist Numeric or NA. Maximum distance threshold for neighbors.
#'   Pairs with distance > filter_dist are not considered neighbors.
#'   Default is NA (no filtering).
#' @param binary Logical. If TRUE (default), return binary adjacency matrix (0/1).
#'   If FALSE, return distance-weighted adjacency matrix.
#' @param verbose Logical. Whether to print progress messages. Default is FALSE.
#'
#' @return A square numeric matrix representing the spatial adjacency/weight matrix.
#'   Row and column names correspond to the spatial locations (from rownames of coords).
#'   \itemize{
#'     \item If \code{binary = TRUE}: Values are 1 (neighbors) or 0 (non-neighbors)
#'     \item If \code{binary = FALSE}: Values are Euclidean distances (0 for non-neighbors)
#'   }
#'
#' @details
#' \strong{Delaunay Triangulation:}
#' Creates a network based on Delaunay triangulation, which maximizes the minimum
#' angle of all triangles. This is a natural way to define neighbors in 2D/3D space.
#' Requires the \code{geometry} package.
#'
#' \strong{K-Nearest Neighbors:}
#' Connects each point to its k nearest neighbors based on Euclidean distance.
#' More robust to irregular spatial distributions but requires choosing k.
#' Requires the \code{RANN} package.
#'
#' @examples
#' # Generate example coordinates
#' set.seed(42)
#' coords <- cbind(x = runif(100), y = runif(100))
#' rownames(coords) <- paste0("spot_", 1:100)
#'
#' \donttest{
#' # Build network using Delaunay (requires geometry package)
#' if (requireNamespace("geometry", quietly = TRUE)) {
#'     W_delaunay <- buildSpatialNetwork(coords, method = "delaunay")
#' }
#'
#' # Build network using KNN (requires RANN package)
#' if (requireNamespace("RANN", quietly = TRUE)) {
#'     W_knn <- buildSpatialNetwork(coords, method = "knn", k = 6)
#' }
#' }
#'
#' @seealso
#' \code{\link{getSpatialNeighbors_Delaunay}}, \code{\link{getSpatialNeighbors_KNN}}
#'
#' @export
buildSpatialNetwork <- function(coords,
                                 method = c("delaunay", "knn"),
                                 k = 10L,
                                 filter_dist = NA,
                                 binary = TRUE,
                                 verbose = FALSE) {

    method <- match.arg(method)

    # Validate inputs
    if (!is.matrix(coords)) {
        coords <- as.matrix(coords)
    }

    if (ncol(coords) < 2) {
        stop("coords must have at least 2 columns (x, y coordinates)")
    }

    if (is.null(rownames(coords))) {
        rownames(coords) <- seq_len(nrow(coords))
    }

    # Build network based on method
    W <- switch(method,
        "delaunay" = getSpatialNeighbors_Delaunay(
            coords,
            filter_dist = filter_dist,
            binary = binary,
            verbose = verbose
        ),
        "knn" = getSpatialNeighbors_KNN(
            coords,
            k = k,
            binary = binary,
            verbose = verbose
        )
    )

    return(W)
}


#' Build Spatial Network via Delaunay Triangulation
#'
#' @description
#' Constructs a spatial adjacency matrix using Delaunay triangulation.
#' Two points are considered neighbors if they share an edge in the triangulation.
#'
#' @param coords Numeric matrix of spatial coordinates.
#'   Rows are spatial locations, columns are x, y (and optionally z) coordinates.
#' @param filter_dist Numeric or NA. Maximum distance threshold for neighbors.
#'   Default is NA (no filtering).
#' @param binary Logical. If TRUE (default), return binary adjacency matrix.
#' @param verbose Logical. Whether to print progress messages. Default is FALSE.
#'
#' @return Square numeric matrix of spatial adjacency weights.
#'
#' @details
#' The function uses Delaunay triangulation from the \code{geometry} package.
#' For 2D coordinates, this creates triangles. For 3D, it creates tetrahedra.
#'
#' Duplicate coordinates are slightly jittered to avoid computational issues.
#'
#' @examples
#' set.seed(42)
#' coords <- cbind(x = runif(50), y = runif(50))
#' rownames(coords) <- paste0("spot_", 1:50)
#'
#' \donttest{
#' if (requireNamespace("geometry", quietly = TRUE)) {
#'     W <- getSpatialNeighbors_Delaunay(coords)
#' }
#' }
#'
#' @export
getSpatialNeighbors_Delaunay <- function(coords,
                                          filter_dist = NA,
                                          binary = TRUE,
                                          verbose = FALSE) {

    # Check for geometry package
    if (!requireNamespace("geometry", quietly = TRUE)) {
        stop("Package 'geometry' is required for Delaunay triangulation. ",
             "Please install it with: install.packages('geometry')")
    }

    # Setup
    if (is.null(rownames(coords))) {
        rownames(coords) <- seq_len(nrow(coords))
    }

    # Handle duplicates by adding small jitter
    if (sum(duplicated(coords)) > 0) {
        if (verbose) {
            message(sprintf("Found %d duplicate coordinates, adding small jitter...",
                          sum(duplicated(coords))))
        }
        coords[duplicated(coords), ] <- coords[duplicated(coords), ] + 1e-6
    }

    # Perform Delaunay triangulation
    tc <- geometry::delaunayn(coords, output.options = FALSE)

    # Extract neighbor pairs based on dimensionality
    if (ncol(coords) == 2) {
        # 2D: triangles have 3 vertices, extract all edges
        ni <- rbind(tc[, c(1, 2)], tc[, c(2, 3)], tc[, c(1, 3)])
    } else if (ncol(coords) == 3) {
        # 3D: tetrahedra have 4 vertices, extract all edges
        nii <- rbind(tc[, c(1, 2, 3)], tc[, c(1, 2, 4)],
                     tc[, c(1, 3, 4)], tc[, c(2, 3, 4)])
        ni <- rbind(nii[, c(1, 2)], nii[, c(2, 3)], nii[, c(1, 3)])
    } else {
        stop("Delaunay triangulation only supports 2D or 3D coordinates")
    }

    # Remove duplicate edges
    ni <- unique(ni)

    # Initialize adjacency matrix
    N <- nrow(coords)
    D <- matrix(0, N, N)
    rownames(D) <- colnames(D) <- rownames(coords)

    # Calculate Euclidean distances for each edge
    for (i in seq_len(nrow(ni))) {
        p1 <- ni[i, 1]
        p2 <- ni[i, 2]
        dist_val <- sqrt(sum((coords[p1, ] - coords[p2, ])^2))

        # Symmetric assignment
        D[p1, p2] <- dist_val
        D[p2, p1] <- dist_val
    }

    # Filter by distance if specified
    if (!is.na(filter_dist)) {
        if (verbose) {
            message(sprintf("Filtering edges with distance > %g", filter_dist))
        }
        D[D > filter_dist] <- 0
    }

    # Binarize if requested
    if (binary) {
        D[D > 0] <- 1
    }

    if (verbose) {
        n_edges <- sum(D > 0) / 2
        message(sprintf("Built Delaunay network: %d nodes, %d edges", N, n_edges))
    }

    return(D)
}


#' Build Spatial Network via K-Nearest Neighbors
#'
#' @description
#' Constructs a spatial adjacency matrix using K-nearest neighbors.
#' Each point is connected to its k nearest neighbors based on Euclidean distance.
#'
#' @param coords Numeric matrix of spatial coordinates.
#' @param k Integer. Number of nearest neighbors. Default is 10.
#' @param mutual Logical. If TRUE, only mutual nearest neighbors are connected
#'   (both A->B and B->A must exist). Default is FALSE.
#' @param binary Logical. If TRUE (default), return binary adjacency matrix.
#'   If FALSE, return distance-weighted matrix.
#' @param verbose Logical. Whether to print progress messages. Default is FALSE.
#'
#' @return Square numeric matrix of spatial adjacency weights.
#'
#' @details
#' Uses the RANN package for efficient nearest neighbor search with KD-trees.
#' The resulting network may be asymmetric (A is neighbor of B doesn't mean
#' B is neighbor of A) unless \code{mutual = TRUE}.
#'
#' @examples
#' set.seed(42)
#' coords <- cbind(x = runif(50), y = runif(50))
#' rownames(coords) <- paste0("spot_", 1:50)
#'
#' \donttest{
#' if (requireNamespace("RANN", quietly = TRUE)) {
#'     W <- getSpatialNeighbors_KNN(coords, k = 6)
#' }
#' }
#'
#' @export
getSpatialNeighbors_KNN <- function(coords,
                                     k = 10L,
                                     mutual = FALSE,
                                     binary = TRUE,
                                     verbose = FALSE) {

    # Check for RANN package
    if (!requireNamespace("RANN", quietly = TRUE)) {
        stop("Package 'RANN' is required for KNN network. ",
             "Please install it with: install.packages('RANN')")
    }

    # Setup
    if (!is.matrix(coords)) {
        coords <- as.matrix(coords)
    }

    if (is.null(rownames(coords))) {
        rownames(coords) <- seq_len(nrow(coords))
    }

    N <- nrow(coords)

    # Validate k
    if (k >= N) {
        warning(sprintf("k (%d) >= number of points (%d), setting k = %d",
                       k, N, N - 1))
        k <- N - 1
    }

    # Find k+1 nearest neighbors (includes self)
    nn_result <- RANN::nn2(coords, k = k + 1)
    nn_idx <- nn_result$nn.idx[, -1, drop = FALSE]  # Remove self (first column)
    nn_dist <- nn_result$nn.dists[, -1, drop = FALSE]

    # Build adjacency matrix
    adj <- matrix(0, N, N)
    rownames(adj) <- colnames(adj) <- rownames(coords)

    for (i in seq_len(N)) {
        neighbors <- nn_idx[i, ]
        distances <- nn_dist[i, ]

        if (binary) {
            adj[i, neighbors] <- 1
        } else {
            adj[i, neighbors] <- distances
        }
    }

    # Make symmetric if mutual neighbors requested
    if (mutual) {
        # Keep only mutual neighbors
        adj <- adj * t(adj)
        if (!binary) {
            # For distance matrix, take the average
            adj <- (adj + t(adj)) / 2
        }
    } else {
        # Make symmetric by union (either direction counts)
        if (binary) {
            adj <- pmax(adj, t(adj))
        } else {
            adj <- (adj + t(adj)) / 2
            adj[adj == 0] <- pmax(adj[adj == 0], t(adj)[adj == 0])
        }
    }

    if (verbose) {
        n_edges <- sum(adj > 0) / 2
        message(sprintf("Built KNN network (k=%d): %d nodes, %d edges", k, N, n_edges))
    }

    return(adj)
}


#' Convert Adjacency Matrix to Edge List
#'
#' @description
#' Converts a spatial adjacency matrix to an edge list data frame.
#' Useful for integration with graph-based methods.
#'
#' @param adj_matrix Square adjacency matrix.
#' @param directed Logical. If FALSE (default), only return unique undirected edges.
#'
#' @return Data frame with columns:
#'   \itemize{
#'     \item \code{from}: Source node name
#'     \item \code{to}: Target node name
#'     \item \code{weight}: Edge weight (if not binary)
#'   }
#'
#' @keywords internal
adj_to_edgelist <- function(adj_matrix, directed = FALSE) {

    # Find non-zero entries
    edges <- which(adj_matrix > 0, arr.ind = TRUE)

    # Create edge list
    edge_df <- data.frame(
        from = rownames(adj_matrix)[edges[, 1]],
        to = colnames(adj_matrix)[edges[, 2]],
        weight = adj_matrix[edges],
        stringsAsFactors = FALSE
    )

    # Remove duplicate edges for undirected network
    if (!directed) {
        # Keep only edges where from < to (lexicographically)
        keep <- edge_df$from < edge_df$to
        edge_df <- edge_df[keep, ]
    }

    return(edge_df)
}


#' Row Standardize Adjacency Matrix
#'
#' @description
#' Performs row standardization of a spatial weights matrix.
#' Each row is divided by its sum, so that the weights in each row sum to 1.
#'
#' @param W Spatial weights matrix.
#'
#' @return Row-standardized weights matrix.
#'
#' @details
#' Row standardization is commonly used in spatial autocorrelation statistics.
#' Rows with zero sum are left unchanged to avoid division by zero.
#'
#' @keywords internal
row_standardize <- function(W) {
    row_sums <- rowSums(W)
    row_sums[row_sums == 0] <- 1  # Avoid division by zero
    W_std <- W / row_sums
    return(W_std)
}
