% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filters.R
\name{filter_emd}
\alias{filter_emd}
\title{Empirical Mode Decomposition Filter}
\usage{
filter_emd(
  y,
  boundary = "periodic",
  max_imf = NULL,
  stop_rule = "type1",
  tol = NULL,
  max_sift = 20,
  verbose = FALSE
)
}
\arguments{
\item{y}{Numeric vector of the time series to decompose.}

\item{boundary}{Character string specifying boundary handling: "periodic"
(default), "symmetric", "none", or "wave".}

\item{max_imf}{Maximum number of IMFs to extract. If NULL, extraction
continues until the residue is monotonic.}

\item{stop_rule}{Character string specifying the stopping criterion for
sifting: "type1" (default), "type2", "type3", "type4", or "type5".}

\item{tol}{Tolerance for sifting convergence. Default is
\code{sd(y) * 0.1^2}.}

\item{max_sift}{Maximum number of sifting iterations per IMF. Default is 20.}

\item{verbose}{Logical indicating whether to print diagnostic messages.}
}
\value{
A list of class "signaly_emd" containing:
\describe{
\item{trend}{Numeric vector of the extracted trend (original minus
residue)}
\item{residue}{Numeric vector of the EMD residue (monotonic trend)}
\item{imfs}{Matrix where each column is an IMF, ordered from highest to
lowest frequency}
\item{n_imfs}{Number of IMFs extracted}
\item{original}{Original input series}
\item{settings}{List of parameters used}
\item{diagnostics}{List with IMF statistics}
}
}
\description{
Applies Empirical Mode Decomposition (EMD) to extract intrinsic mode
functions (IMFs) from a time series. Unlike Fourier or wavelet methods,
EMD is fully data-adaptive and does not require pre-specified basis
functions, making it suitable for non-stationary and non-linear signals.
}
\details{
EMD decomposes a signal x(t) into a sum of Intrinsic Mode Functions (IMFs)
and a residue:
\deqn{x(t) = \sum_{j=1}^{n} c_j(t) + r_n(t)}

where each IMF \eqn{c_j(t)} satisfies two conditions:
\enumerate{
\item The number of extrema and zero crossings differ by at most one
\item The mean of upper and lower envelopes is zero at each point
}

The sifting process iteratively extracts IMFs from highest to lowest
frequency until the residue becomes monotonic (representing the trend).
}
\section{Advantages over Fourier/Wavelet Methods}{

\itemize{
\item \strong{Adaptive basis}: IMFs are derived from the data itself, not
pre-specified
\item \strong{Handles non-stationarity}: Instantaneous frequency can vary
over time
\item \strong{Handles non-linearity}: No assumption of linear superposition
\item \strong{Preserves local structure}: Better time localization than
Fourier methods
}
}

\section{Limitations}{

\itemize{
\item \strong{Mode mixing}: Different scales may appear in the same IMF
\item \strong{End effects}: Boundary conditions can cause artifacts
\item \strong{No formal theory}: Unlike wavelets, lacks rigorous mathematical
foundation
\item \strong{Reproducibility}: Results can vary with stopping criteria
}
}

\examples{
set.seed(123)
t <- seq(0, 10, length.out = 200)
y <- sin(2*pi*t) + 0.5*sin(8*pi*t) + 0.1*rnorm(200)
result <- filter_emd(y)
plot(y, type = "l", col = "gray")
lines(result$trend, col = "red", lwd = 2)

}
\references{
Huang, N. E., Shen, Z., Long, S. R., Wu, M. C., Shih, H. H., Zheng, Q.,
Yen, N.-C., Tung, C. C., & Liu, H. H. (1998). The empirical mode
decomposition and the Hilbert spectrum for nonlinear and non-stationary
time series analysis. Proceedings of the Royal Society A, 454(1971), 903-995.

Wu, Z., & Huang, N. E. (2009). Ensemble empirical mode decomposition: A
noise-assisted data analysis method. Advances in Adaptive Data Analysis,
1(1), 1-41.
}
\seealso{
\code{\link[EMD]{emd}}, \code{\link{filter_wavelet}},
\code{\link{filter_hpgc}}
}
