% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/signal_analysis.R
\name{signal_analysis}
\alias{signal_analysis}
\title{Comprehensive Signal Analysis for Panel Data}
\usage{
signal_analysis(
  data,
  y_formula,
  time_var = NULL,
  group_var = NULL,
  methods = "all",
  filter_config = list(),
  horseshoe_config = list(),
  pca_config = list(),
  dfm_config = list(),
  unitroot_tests = "all",
  na_action = c("interpolate", "omit", "fail"),
  standardize = TRUE,
  first_difference = FALSE,
  verbose = TRUE,
  seed = NULL
)
}
\arguments{
\item{data}{A data.frame or matrix containing the panel data. For data.frames,
time should be in rows and variables in columns.}

\item{y_formula}{Formula specifying how to construct Y from X variables, or
a character string naming the pre-constructed Y column in data.}

\item{time_var}{Character string naming the time variable (optional, assumes
rows are ordered by time if NULL).}

\item{group_var}{Character string naming the group/panel variable for panel
data (optional for single time series).}

\item{methods}{Character vector specifying which methods to apply. Options:
\code{"wavelet"}, \code{"emd"}, \code{"hpgc"}, \code{"horseshoe"},
\code{"pca"}, \code{"dfm"}, \code{"unitroot"}, or \code{"all"} (default).}

\item{filter_config}{List of configuration options for filtering methods:
\describe{
\item{wavelet_filter}{Wavelet filter type (default: "la8")}
\item{wavelet_levels}{Which detail levels to combine (default: c(3,4))}
\item{emd_max_imf}{Maximum IMFs for EMD (default: 10)}
\item{hpgc_prior}{Prior configuration: "weak", "informative", "empirical" (default: "weak")}
\item{hpgc_chains}{Number of MCMC chains (default: 4)}
\item{hpgc_iterations}{Total iterations per chain (default: 20000)}
}}

\item{horseshoe_config}{List of configuration for Horseshoe regression:
\describe{
\item{p0}{Expected number of relevant predictors (default: NULL for auto)}
\item{chains}{Number of MCMC chains (default: 4)}
\item{iter_sampling}{Sampling iterations per chain (default: 2000)}
\item{iter_warmup}{Warmup iterations (default: 1000)}
\item{adapt_delta}{Target acceptance rate (default: 0.95)}
\item{use_qr}{Use QR decomposition (default: TRUE)}
\item{kappa_threshold}{Shrinkage threshold for selection (default: 0.5)}
}}

\item{pca_config}{List of configuration for PCA:
\describe{
\item{n_components}{Number of components (default: NULL for auto)}
\item{rotation}{Rotation method: "none", "varimax", "oblimin" (default: "none")}
\item{n_boot}{Bootstrap replications (default: 1000)}
\item{block_length}{Block length for bootstrap (default: NULL for auto)}
\item{alpha}{Alpha for bootstrap tests (default: 0.05)}
}}

\item{dfm_config}{List of configuration for Dynamic Factor Models:
\describe{
\item{r}{Number of factors (default: NULL for auto via IC)}
\item{max_factors}{Maximum factors to consider (default: 10)}
\item{p}{VAR lags for factor dynamics (default: 1)}
\item{ic}{Information criterion: "IC1", "IC2", "IC3" (default: "bai_ng_2")}
}}

\item{unitroot_tests}{Character vector of unit root tests to apply. Options:
\code{"adf"}, \code{"ers"}, \code{"kpss"}, \code{"pp"}, or \code{"all"} (default).}

\item{na_action}{How to handle missing values: "interpolate", "omit", "fail" (default: "interpolate").}

\item{standardize}{Logical, whether to standardize variables before analysis (default: TRUE).}

\item{first_difference}{Logical, whether to first-difference data (default: FALSE).}

\item{verbose}{Logical, whether to print progress messages (default: TRUE).}

\item{seed}{Random seed for reproducibility (default: NULL).}
}
\value{
An S3 object of class \code{"signal_analysis"} containing:
\describe{
\item{call}{The matched function call}
\item{data}{Processed input data}
\item{Y}{The constructed target signal}
\item{X}{The predictor matrix}
\item{filters}{Results from spectral decomposition methods}
\item{horseshoe}{Results from Bayesian variable selection}
\item{pca}{Results from PCA with bootstrap}
\item{dfm}{Results from Dynamic Factor Model}
\item{unitroot}{Results from unit root tests}
\item{interpretation}{Automated technical interpretation}
\item{config}{Configuration parameters used}
}
}
\description{
Master function that orchestrates the complete signal extraction pipeline,
integrating spectral decomposition (wavelets, EMD, HP-GC), Bayesian variable' selection (regularized Horseshoe), dimensionality reduction (PCA, DFM), and
stationarity testing into a unified analytical framework.

The function constructs a target signal Y from candidate variables X in panel
data and applies multiple complementary methodologies to extract the latent
structure from phenomenological dynamics.
}
\details{
\strong{Methodological Framework}

The signal extraction pipeline distinguishes between latent structure
(the underlying data-generating process) and phenomenological dynamics
(observed variability). This is achieved through:

\enumerate{
\item \strong{Spectral Decomposition}: Separates signal frequencies
\itemize{
\item Wavelets: Multi-resolution analysis via MODWT
\item EMD: Data-adaptive decomposition into intrinsic modes
\item HP-GC: Bayesian unobserved components (trend + cycle)
}
\item \strong{Sparse Regression}: Identifies relevant predictors
\itemize{
\item Regularized Horseshoe: Adaptive shrinkage with slab regularization
\item Shrinkage factors (kappa) quantify predictor relevance
}
\item \strong{Dimensionality Reduction}: Extracts common factors
\itemize{
\item PCA: Static factor structure with bootstrap significance
\item DFM: Dynamic factors with VAR transition dynamics
}
\item \strong{Stationarity Testing}: Characterizes persistence properties
\itemize{
\item Integrated battery of ADF, ERS, KPSS, PP tests
\item Synthesized conclusion on stationarity type
}
}

\strong{Interpretation Framework}

The automated interpretation assesses:
\itemize{
\item \strong{Signal Smoothness}: Variance of second differences
\item \strong{Trend Persistence}: Deterministic vs. stochastic via unit roots
\item \strong{Information Topology}: Entropy of PC1 loadings (concentrated vs. diffuse)
\item \strong{Sparsity Ratio}: Proportion of predictors shrunk to zero
\item \strong{Factor Structure}: Number of significant common factors
}
}
\examples{
\donttest{
# Generate example panel data
set.seed(42)
n_time <- 50   
n_vars <- 10   

# Create correlated predictors with common factor structure
factors <- matrix(rnorm(n_time * 2), n_time, 2)
loadings <- matrix(runif(n_vars * 2, -1, 1), n_vars, 2)
X <- factors \%*\% t(loadings) + matrix(rnorm(n_time * n_vars, 0, 0.5), n_time, n_vars)
colnames(X) <- paste0("X", 1:n_vars)

# True signal depends on only 3 predictors
true_beta <- c(rep(1, 3), rep(0, 7))
Y <- X \%*\% true_beta + rnorm(n_time, 0, 0.5)

# Combine into data frame
data <- data.frame(Y = Y, X)

# Run comprehensive analysis
# We pass specific configs to make MCMC very fast just for the example
result <- signal_analysis(
  data = data,
  y_formula = "Y",
  methods = "all",
  verbose = TRUE,
  # Configuration for speed (CRAN policy < 5s preferred)
  filter_config = list(
     hpgc_chains = 1,      
     hpgc_iterations = 50, 
     hpgc_burnin = 10
  ),
  horseshoe_config = list(
     chains = 1,           
     iter_sampling = 50,   
     iter_warmup = 10
  ),
  pca_config = list(
     n_boot = 50           
  )
)

# View interpretation
print(result)

# Plot results
plot(result)
}

}
\references{
Piironen, J., & Vehtari, A. (2017). Sparsity information and regularization
in the horseshoe and other shrinkage priors. Electronic Journal of Statistics,
11(2), 5018-5051. \doi{10.1214/17-EJS1337SI}

Bai, J., & Ng, S. (2002). Determining the Number of Factors in Approximate
Factor Models. Econometrica, 70(1), 191-221. \doi{10.1111/1468-0262.00273}
}
\seealso{
\code{\link{filter_wavelet}}, \code{\link{filter_emd}}, \code{\link{filter_hpgc}},
\code{\link{fit_horseshoe}}, \code{\link{pca_bootstrap}}, \code{\link{estimate_dfm}},
\code{\link{test_unit_root}}
}
