#' prop_target
#'
#' Calculates the proportion of normalized survivorship \eqn{S(L)} falling inside
#' harvest slot limits \code{[minLS, maxLS]} relative to the exploitable population
#' (>\code{L_c}), where \eqn{S(L) = \exp(-M \, t(L))}
#' and \eqn{t(L)} is the inverse age-from-length for a chosen growth model.
#'
#' Supported growth models (reparameterized to avoid negative length-at-age-0
#' and to give exact \eqn{t(L_{start})=0}):
#' \itemize{
#'   \item \strong{von Bertalanffy (VB)} with start length \eqn{l_0}:
#'   \deqn{t(l) = -\frac{1}{K}\ln\!\left(\frac{L_\infty - l}{L_\infty - l_0}\right), \quad
#'         l(t) = L_\infty\!\left(1 - (1-l_0/L_\infty)\,e^{-Kt}\right).}
#'   \item \strong{Gompertz} with start length \eqn{l_0} (requires \eqn{0 < l_0 < L_\infty}):
#'   \deqn{t(l) = -\frac{1}{K}\ln\!\left(\frac{\ln(l/L_\infty)}{\ln(l_0/L_\infty)}\right), \quad
#'         l(t) = L_\infty\,(l_0/L_\infty)^{e^{-Kt}}.}
#'   \item \strong{Schnute} with \eqn{l(0)=0} and \eqn{l(t_{max})=l_2}:
#'   \deqn{t(l) = -\frac{1}{g_1}\ln\!\left(1 - \frac{l^{g_2}}{l_2^{g_2}}\,(1-e^{-g_1 t_{max}})\right), \quad
#'         l(t) = \left(\frac{l_2^{g_2}}{1-e^{-g_1 t_{max}}}\,(1-e^{-g_1 t})\right)^{1/g_2}.}
#' }
#'
#' Survivorship is normalized at the model start so that \eqn{S(L_{start})=1}:
#' \code{l0} for vB, \code{Gom_l0} for Gompertz (requires \code{0 < Gom_l0 < Gom_Linf}),
#' and \eqn{0} for Schnute.
#'
#' @param minLS,maxLS Numeric. Minimum and maximum harvest slot limits (same units as length).
#' @param Lc Numeric. Lower cutoff; individuals below \code{Lc} are \emph{not exploitable}.
#' @param M Numeric or \code{NULL}. Natural mortality. If \code{NULL},
#'   defaults to \eqn{M = 4.899\,t_{max}^{-0.916}}.
#' @param growth_model Character. One of \code{"vb"}, \code{"gompertz"}, \code{"schnute"}.
#' @param Linf,K,l0 VB parameters; \code{l0} is the start length (default 0).
#' @param tmax Numeric. Maximum age used to determine \eqn{l(t_{max})} and set the upper integration bound.
#' @param Gom_Linf,Gom_K,Gom_l0 Gompertz parameters; requires \code{0 < Gom_l0 < Gom_Linf}.
#' @param g1,g2,l2 Schnute parameters; \code{l2 = l(t_{max})}; requires \code{g1 > 0}, \code{l2 > 0}, \code{g2 != 0}.
#' @param Lmin Optional numeric. Lower bound for the curve grid. If \code{NULL} it uses the
#'   model’s start length (\code{l0}, \code{Gom_l0}, or \code{0}).
#' @param plot Logical. If \code{TRUE}, return a \pkg{ggplot2} visual; default \code{FALSE}.
#' @param length_units Optional character scalar. Units to display in the x-axis
#'   label when `plot = TRUE` (e.g., `"cm"` or `"mm"`). If `NULL` (default),
#'   the label is simply `"Length"`.
#'
#' @details
#' Targeted proportion:
#' \deqn{\frac{\int_{\max(minLS,L_c)}^{\min(maxLS,\,l(t_{max}))} S(L)\,dL}{
#'              \int_{\max(L_c,\,L_{start})}^{l(t_{max})} S(L)\,dL}.}
#' We clamp only near the upper limit to avoid \code{log(0)} and never shift the start,
#' preserving \eqn{t(L_{start})=0}.
#'
#' @return
#' If \code{plot = FALSE} (default): numeric scalar (the targeted proportion).
#' If \code{plot = TRUE}: list with \code{proportion} and \code{plot} (a ggplot object).
#'
#' @examples
#' # Numeric only
#' prop_target(minLS=120, maxLS=240, Lc=80,
#'   growth_model="vb", Linf=405, K=0.118, l0=0, tmax=34, plot=FALSE)
#'
#' \donttest{
#' # With plot (requires ggplot2)
#' out <- prop_target(minLS=120, maxLS=240, Lc=80,
#'   growth_model="schnute", g1=0.2, g2=0.2, l2=405, tmax=34, plot=TRUE, length_units = "mm")
#' out$plot
#' }
#'
#' @export
prop_target <- function(
    minLS = NULL, maxLS = NULL, Lc = NULL, M = NULL,
    growth_model = c("vb", "gompertz", "schnute"),
    Linf = NULL, K = NULL, l0 = 0, tmax = NULL,
    Gom_Linf = NULL, Gom_K = NULL, Gom_l0 = NULL,
    g1 = NULL, g2 = NULL, l2 = NULL,
    Lmin = NULL,
    plot = FALSE,
    length_units = NULL
) {
  growth_model <- match.arg(growth_model)
  if (!is.numeric(minLS) || !is.numeric(maxLS) || !is.numeric(Lc))
    stop("`minLS`, `maxLS`, and `Lc` must be numeric.", call. = FALSE)
  if (!is.finite(minLS) || !is.finite(maxLS) || !is.finite(Lc))
    stop("`minLS`, `maxLS`, and `Lc` must be finite.", call. = FALSE)
  if (maxLS <= minLS)
    stop("Require maxLS > minLS.", call. = FALSE)
  if (tmax <= 0 || !is.finite(tmax))
    stop("`tmax` must be a positive, finite number.", call. = FALSE)

  if (is.null(M)) M <- 4.899 * tmax^-0.916
  if (!is.finite(M) || M <= 0)
    stop("`M` must be positive and finite.", call. = FALSE)

  if (!is.null(length_units)) {
    if (!is.character(length_units) || length(length_units) != 1L ||
        is.na(length_units) || !nzchar(trimws(length_units))) {
      stop("`length_units` must be a non-empty character scalar or NULL.", call. = FALSE)
    }
  }

  if (!is.null(Lmin)) {
    if (!is.numeric(Lmin) || length(Lmin) != 1L || !is.finite(Lmin)) {
      stop("`Lmin` must be a single finite numeric or NULL.", call. = FALSE)
    }
  }

  eps <- 1e-7

  # --- model validation ---
  if (growth_model == "vb") {
    if (is.null(Linf) || is.null(K)) stop("VB: `Linf` and `K` must be provided.", call. = FALSE)
    if (Linf <= 0 || K <= 0)       stop("VB: `Linf` and `K` must be positive.", call. = FALSE)
    if (l0 < 0 || l0 >= Linf)      stop("VB: require 0 <= l0 < Linf.", call. = FALSE)
  } else if (growth_model == "gompertz") {
    if (is.null(Gom_Linf) || is.null(Gom_K)) stop("Gompertz: `Gom_Linf` and `Gom_K` must be provided.", call. = FALSE)
    if (Gom_Linf <= 0 || Gom_K <= 0)         stop("Gompertz: `Gom_Linf` and `Gom_K` must be positive.", call. = FALSE)
    if (is.null(Gom_l0)) stop("Gompertz: `Gom_l0` must be provided and satisfy 0 < Gom_l0 < Gom_Linf.", call. = FALSE)
    if (Gom_l0 <= 0 || Gom_l0 >= Gom_Linf)   stop("Gompertz: require 0 < Gom_l0 < Gom_Linf.", call. = FALSE)
  } else { # schnute
    if (is.null(g1) || is.null(g2) || is.null(l2)) stop("Schnute: `g1`, `g2`, and `l2` must be provided.", call. = FALSE)
    if (g1 <= 0)       stop("Schnute: require g1 > 0.", call. = FALSE)
    if (l2 <= 0)       stop("Schnute: require l2 > 0.", call. = FALSE)
    if (abs(g2) < .Machine$double.eps) stop("Schnute: require g2 != 0 for this parameterization.", call. = FALSE)
  }

  # Start-length anchor
  anchor_L <- switch(growth_model,
                     "vb"       = l0,
                     "gompertz" = Gom_l0,
                     "schnute"  = 0
  )
  if (is.null(Lmin)) Lmin <- anchor_L

  # Inverse age-from-length (vectorized)
  age_from_length <- switch(
    growth_model,
    "vb" = {
      function(L) {
        L_eff <- pmin(pmax(L, l0), Linf - eps)
        val   <- (Linf - L_eff) / (Linf - l0)
        age   <- -log(pmax(val, eps)) / K
        age[L <= l0] <- 0
        pmin(age, tmax)
      }
    },
    "gompertz" = {
      denom_ln <- log(Gom_l0 / Gom_Linf) # < 0
      function(L) {
        L_eff     <- pmin(pmax(L, Gom_l0), Gom_Linf - eps)
        log_ratio <- log(L_eff / Gom_Linf) / denom_ln
        age       <- -log(pmax(log_ratio, eps)) / Gom_K
        age[L <= Gom_l0] <- 0
        pmin(age, tmax)
      }
    },
    "schnute" = {
      denom <- (1 - exp(-g1 * tmax))
      function(L) {
        L_eff  <- pmin(pmax(L, 0), l2 - eps)
        inside <- 1 - ((L_eff^g2) / (l2^g2)) * denom
        inside <- pmin(pmax(inside, eps), 1)
        age <- -log(inside) / g1
        age[L <= 0] <- 0
        pmin(age, tmax)
      }
    }
  )

  # Survivorship normalized at model start (compute S0 once)
  S0 <- exp(-M * age_from_length(anchor_L))
  S_norm <- function(L) {
    age <- age_from_length(L)
    val <- exp(-M * age) / S0
    ifelse(is.finite(val), val, 0)
  }

  # Length at tmax
  length_at_t_max <- switch(
    growth_model,
    "vb"       = Linf * (1 - (1 - l0 / Linf) * exp(-K * tmax)),
    "gompertz" = Gom_Linf * ((Gom_l0 / Gom_Linf) ^ (exp(-Gom_K * tmax))),
    "schnute"  = l2
  )

  # Integration limits
  lower_den <- max(Lc, Lmin)
  upper_den <- max(length_at_t_max, lower_den + eps)
  lower_num <- max(minLS, Lc)
  upper_num <- min(maxLS, length_at_t_max)

  # Short-circuit if no overlap
  if (upper_num <= lower_num) {
    warning("No overlap between [minLS, maxLS] and exploitable range [Lc, l(tmax)]; returning 0.", call. = FALSE)
    proportion <- 0
  } else {
    # Safer integrate with rel.tol; catch numerical failures gracefully
    total_exploitable <- try(stats::integrate(S_norm, lower = lower_den, upper = upper_den, rel.tol = 1e-6)$value, silent = TRUE)
    in_slot           <- try(stats::integrate(S_norm, lower = lower_num, upper = upper_num, rel.tol = 1e-6)$value, silent = TRUE)

    if (inherits(total_exploitable, "try-error") || inherits(in_slot, "try-error")) {
      stop("Numerical integration failed; check parameter values and ranges.", call. = FALSE)
    }
    if (total_exploitable <= 0) {
      stop("Total exploitable integral is non-positive; check inputs.", call. = FALSE)
    }
    proportion <- in_slot / total_exploitable
  }

  if (!isTRUE(plot)) return(proportion)

  # --- Optional plot (returned, not printed) ---
  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    warning("Package 'ggplot2' not installed; returning proportion without plot.", call. = FALSE)
    return(list(proportion = proportion, plot = NULL))
  }

  by_step <- max(0.1, (length_at_t_max - anchor_L) / 1000)
  length_values <- seq(anchor_L, length_at_t_max, by = by_step)
  survivorship_values <- exp(-M * age_from_length(length_values))
  normalized_survivorship_values <- survivorship_values / S0

  df <- data.frame(
    Length = length_values,
    Survivorship = normalized_survivorship_values
  )
  df$Category <- "Out-of-slot"
  df$Category[df$Length < Lc] <- "<Lc"
  in_slot_idx <- df$Length >= lower_num & df$Length <= upper_num
  df$Category[in_slot_idx] <- "In-slot"
  df$Category <- factor(df$Category, levels = c("<Lc", "Out-of-slot", "In-slot"))

  title_label <- switch(growth_model, "vb" = "vB", "gompertz" = "Gompertz", "schnute" = "Schnute")
  x_lab <- if (is.null(length_units)) "Length" else paste0("Length (", length_units, ")")

  .use_linewidth <- function() {
    requireNamespace("ggplot2", quietly = TRUE) &&
      utils::packageVersion("ggplot2") >= "3.4.0"
  }

  gg <- ggplot2::ggplot(df, ggplot2::aes(x = Length, y = Survivorship)) +
    ggplot2::geom_ribbon(
      data = df[df$Category == "<Lc", , drop = FALSE],
      ggplot2::aes(ymin = 0, ymax = Survivorship, fill = Category), alpha = 0.30
    ) +
    ggplot2::geom_ribbon(
      data = df[df$Category == "In-slot", , drop = FALSE],
      ggplot2::aes(ymin = 0, ymax = Survivorship, fill = Category), alpha = 0.30
    ) +
    ggplot2::geom_ribbon(
      data = df[df$Category == "Out-of-slot", , drop = FALSE],
      ggplot2::aes(ymin = 0, ymax = Survivorship, fill = Category), alpha = 0.15
    ) +
    ggplot2::geom_line(
      size = if (.use_linewidth()) NULL else 1,
      linewidth = if (.use_linewidth()) 1 else NULL,
      col = "darkgreen"
    ) +
    ggplot2::geom_vline(
      xintercept = minLS,
      linetype = "dashed",
      size = if (.use_linewidth()) NULL else 1,
      linewidth = if (.use_linewidth()) 1 else NULL,
      col = "darkred"
    ) +
    ggplot2::geom_vline(
      xintercept = maxLS,
      linetype = "dashed",
      size = if (.use_linewidth()) NULL else 1,
      linewidth = if (.use_linewidth()) 1 else NULL,
      col = "darkred"
    ) +
    ggplot2::labs(
      title = paste0("Growth model: ", title_label),
      x = x_lab, y = "Normalized survivorship", fill = ""
    ) +
    ggplot2::scale_fill_manual(
      values = c("<Lc" = "grey60", "Out-of-slot" = "gold3", "In-slot" = "yellow"),
      breaks = c("<Lc", "Out-of-slot", "In-slot"),
      labels = c("< Lc", "Out-of-slot", "In-slot")
    ) +
    ggplot2::coord_cartesian(
      ylim = c(0, 1),
      xlim = c(0, max(length_at_t_max, maxLS, Lc)),
      expand = FALSE
    ) +
    ggplot2::theme_bw(base_size = 14) +
    ggplot2::theme(
      plot.title = ggplot2::element_text(hjust = 0.5),
      plot.margin = grid::unit(c(0.25, 0.25, 0.25, 0.25), "cm"),
      text = ggplot2::element_text(size = 15),
      legend.position = "top",
      legend.direction = "horizontal",
      legend.background = ggplot2::element_blank(),
      legend.key = ggplot2::element_blank()
    )

  list(proportion = proportion, plot = gg)
}
