# Bias specification & validation for AppRaise

#' Validate bias selections
#'
#' @param b_types,s_types,d_types,e_types,en_types Character vectors of bias names
#' @param num_biases Integer, total number of biases declared
#'
#' @return Invisibly TRUE, otherwise errors
#' @export
validate_bias_selection <- function(
    b_types, s_types, d_types, e_types, en_types, num_biases
) {
  all_biases <- c(b_types, s_types, d_types, e_types, en_types)
  all_biases <- all_biases[all_biases != ""]

  if (length(all_biases) != num_biases) {
    stop("Number of biases selected does not match num_biases", call. = FALSE)
  }

  if (any(duplicated(all_biases))) {
    stop("A bias cannot be modeled with more than one distribution", call. = FALSE)
  }

  invisible(TRUE)
}


#' Validate positivity constraints
#'
#' @param values Numeric vector
#' @param exceptions Indices allowed to be non-positive
#' @export
validate_positive <- function(values, exceptions = NULL) {
  values <- ifelse(is.na(values), 999, values)

  if (!is.null(exceptions)) {
    values <- values[-exceptions]
  }

  if (!all(values > 0)) {
    stop("Non-positive values found where positive values are required", call. = FALSE)
  }

  invisible(TRUE)
}


#' Build bias specification for Stan
#'
#' @param num_biases Integer. Total number of biases.
#' @param b_types Character vector of biases with beta priors.
#' @param s_types Character vector of biases with skew-normal priors.
#' @param d_types Character vector of biases with Laplace priors.
#' @param e_types Character vector of biases with exponential priors.
#' @param en_types Character vector of biases with negative exponential priors.
#' @param ab_params Named list of beta prior parameters.
#' @param skn_params Named list of skew-normal prior parameters.
#' @param de_params Named list of Laplace prior parameters.
#' @param ex_params Named list of exponential prior parameters.
#' @param exneg_params Named list of negative exponential prior parameters.
#'
#' @return A list defining bias structure and prior parameters.
#'
#' @examples
#' ## Example 1: Single bias with a Beta prior
#'
#' bias_spec <- build_bias_specification(
#'   num_biases = 1,
#'   b_types = "Confounding",
#'   ab_params = list(
#'     Confounding = c(2, 5)
#'   )
#' )
#'
#' bias_spec
#'
#' ## Example 2: Multiple biases with different prior families
#'
#' bias_spec <- build_bias_specification(
#'   num_biases = 2,
#'   b_types = "Confounding",
#'   s_types = "Selection Bias",
#'   ab_params = list(
#'     Confounding = c(2, 5)
#'   ),
#'   skn_params = list(
#'     `Selection Bias` = c(0, 0.2, 5)
#'   )
#' )
#'
#' bias_spec
#'
#' ## Example 3: Exponential bias prior
#'
#' bias_spec <- build_bias_specification(
#'   num_biases = 1,
#'   e_types = "Measurement Errors",
#'   ex_params = list(
#'     `Measurement Errors` = 1.5
#'   )
#' )
#'
#' bias_spec
#'
#' @seealso
#' \itemize{
#'   \item \code{\link{simulate_bias_priors}} for sampling bias prior distributions
#'   \item \code{\link{run_appraise_model}} for posterior inference
#'   \item \code{vignette("appraise-introduction")} for a full workflow
#' }
#'
#' @references
#' Kabali C (2025). AppRaise: Software for quantifying evidence uncertainty
#' in systematic reviews using a posterior mixture model. *Journal of
#' Evaluation in Clinical Practice*, 31, 1-12. https://doi.org/10.1111/jep.70272.
#' @export

build_bias_specification <- function(
    num_biases,
    b_types = character(),
    s_types = character(),
    d_types = character(),
    e_types = character(),
    en_types = character(),
    ab_params = list(),
    skn_params = list(),
    de_params = list(),
    ex_params = list(),
    exneg_params = list()
) {

  bias_map <- appraise::bias_map

  validate_bias_selection(
    b_types, s_types, d_types, e_types, en_types, num_biases
  )

  b  <- rep(999, num_biases)
  s  <- rep(999, num_biases)
  d  <- rep(999, num_biases)
  e  <- rep(999, num_biases)
  en <- rep(999, num_biases)

  ab_values    <- rep(999, 2 * num_biases)
  skn_values   <- rep(999, 3 * num_biases)
  de_values    <- rep(999, 2 * num_biases)
  ex_values    <- rep(999, num_biases)
  exneg_values <- rep(999, num_biases)

  j <- 1

  for (bias in b_types) {
    idx <- bias_map[bias]
    b[j] <- idx
    ab_values[2*j - 1] <- ab_params[[bias]][1]
    ab_values[2*j]     <- ab_params[[bias]][2]
    j <- j + 1
  }

  for (bias in s_types) {
    idx <- bias_map[bias]
    s[j] <- idx
    skn_values[3*j - 2] <- skn_params[[bias]][1]
    skn_values[3*j - 1] <- skn_params[[bias]][2]
    skn_values[3*j]     <- skn_params[[bias]][3]
    j <- j + 1
  }

  for (bias in d_types) {
    idx <- bias_map[bias]
    d[j] <- idx
    de_values[2*j - 1] <- de_params[[bias]][1]
    de_values[2*j]     <- de_params[[bias]][2]
    j <- j + 1
  }

  for (bias in e_types) {
    idx <- bias_map[bias]
    e[j] <- idx
    ex_values[j] <- ex_params[[bias]][1]
    j <- j + 1
  }

  for (bias in en_types) {
    idx <- bias_map[bias]
    en[j] <- idx
    exneg_values[j] <- exneg_params[[bias]][1]
    j <- j + 1
  }

  validate_positive(ab_values)
  validate_positive(skn_values, exceptions = c(1,3,4,6,7,9,10,12,13,15))
  validate_positive(de_values,  exceptions = c(1,3,5,7,9))
  validate_positive(ex_values)
  validate_positive(exneg_values)

  list(
    NN = num_biases,
    b = b,
    s = s,
    d = d,
    e = e,
    en = en,
    ab_values = ab_values,
    skn_values = skn_values,
    de_values = de_values,
    ex_values = ex_values,
    exneg_values = exneg_values
  )
}
