## appraise

**Bias-aware evidence synthesis using posterior mixture models**

## Overview

**appraise** is an R package for bias-aware evidence synthesis in systematic reviews and health technology assessments. It implements the AppRaise framework, which explicitly models multiple sources of bias at the study level and propagates uncertainty using posterior simulation.

Rather than assuming a single pooled likelihood (as in conventional meta-analysis), appraise combines study-specific posterior distributions using posterior mixture models. This allows probabilistic inference on clinically or policy-relevant thresholds while accounting for bias, uncertainty, and variation in study relevance.

The methods are designed to support transparent decision-making when the evidence base is heterogeneous and potentially biased.

## Installation

The package is currently available from GitHub and will be submitted to CRAN.

``` r
# Install from GitHub
# install.packages("remotes")
remotes::install_github("conibyera/appraise")
```

Once on CRAN:

``` r
install.packages("appraise")
```

## Core functionality

The typical workflow in **appraise** consists of four steps:

**1.** Specify bias structures and prior distributions

**2.** Simulate bias priors

**3.** Fit the bias-adjusted model

**4.** Summarize and synthesize posterior distributions

## 1. Bias specification

Biases are specified by type and linked to prior distributions.

``` r
library(appraise)

bias_spec <- build_bias_specification(
  num_biases = 2,
  b_types = "Confounding",
  s_types = "Selection Bias",
  ab_params = list(Confounding = c(2, 5)),
  skn_params = list(`Selection Bias` = c(0, 0.2, 5))
)
```

## 2. Simulate bias priors

``` r
xi <- simulate_bias_priors(bias_spec, n_draws = 2000)
```

## 3. Fit the AppRaise model

``` r
fit <- run_appraise_model(
  bias_spec = bias_spec,
  yhat = -0.6,
  stdev = 0.12,
  threshold_value = -0.4
)
```

## 4. Posterior inference

Compute posterior probabilities and summaries directly from the draws.

``` r
posterior_probability(fit$mid)

posterior_summary(fit$theta)
```

## Evidence synthesis using posterior mixtures

When multiple studies are available, posterior distributions can be combined using mixture models.

``` r
mixture <- posterior_mixture(
  theta_list = list(fit$theta, rnorm(length(fit$theta), -0.4, 0.15)),
  weights = c(0.6, 0.4)
)

mixture$summary
```

## Shiny application

The **AppRaise Shiny application** provides a graphical interface to the same statistical functions implemented in this package. All computations are performed using exported package functions; the app adds interactivity, visualization, and workflow support.

The Shiny app is intended for applied analysts and decision-makers, while the package supports reproducible, script-based workflows.

## Documentation

-   Package vignette:

``` r
vignette("appraise-introduction")
```

## References

Kabali C. (2025). AppRaise: Software for quantifying evidence uncertainty in systematic reviews using a posterior mixture model. *Journal of Evaluation in Clinical Practice*, 31, 1-12. <https://doi.org/10.1111/jep.70272>

## License

GPL-3
