% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/betaMC-mc-mi.R
\name{MCMI}
\alias{MCMI}
\title{Generate the Sampling Distribution of Regression Parameters
Using the Monte Carlo Method for Data with Missing Values}
\usage{
MCMI(
  object,
  mi,
  R = 20000L,
  type = "hc3",
  g1 = 1,
  g2 = 1.5,
  k = 0.7,
  decomposition = "eigen",
  pd = TRUE,
  tol = 1e-06,
  fixed_x = FALSE,
  seed = NULL
)
}
\arguments{
\item{object}{Object of class \code{lm}.}

\item{mi}{Object of class \code{mids} (output of \code{\link[mice:mice]{mice::mice()}}),
object of class \code{amelia} (output of \code{\link[Amelia:amelia]{Amelia::amelia()}}),
or a list of multiply imputed data sets.}

\item{R}{Positive integer.
Number of Monte Carlo replications.}

\item{type}{Character string.
Sampling covariance matrix type.
Possible values are
\code{"mvn"},
\code{"adf"},
\code{"hc0"},
\code{"hc1"},
\code{"hc2"},
\code{"hc3"},
\code{"hc4"},
\code{"hc4m"}, and
\code{"hc5"}.
\code{type = "mvn"} uses the normal-theory sampling covariance matrix.
\code{type = "adf"} uses the asymptotic distribution-free
sampling covariance matrix.
\code{type = "hc0"} through \code{"hc5"} uses different versions of
heteroskedasticity-consistent sampling covariance matrix.}

\item{g1}{Numeric.
\code{g1} value for \code{type = "hc4m"}.}

\item{g2}{Numeric.
\code{g2} value for \code{type = "hc4m"}.}

\item{k}{Numeric.
Constant for \code{type = "hc5"}}

\item{decomposition}{Character string.
Matrix decomposition of the sampling variance-covariance matrix
for the data generation.
If \code{decomposition = "chol"}, use Cholesky decomposition.
If \code{decomposition = "eigen"}, use eigenvalue decomposition.
If \code{decomposition = "svd"}, use singular value decomposition.}

\item{pd}{Logical.
If \code{pd = TRUE},
check if the sampling variance-covariance matrix
is positive definite using \code{tol}.}

\item{tol}{Numeric.
Tolerance used for \code{pd}.}

\item{fixed_x}{Logical.
If \code{fixed_x = TRUE}, treat the regressors as fixed.
If \code{fixed_x = FALSE}, treat the regressors as random.}

\item{seed}{Integer.
Seed number for reproducibility.}
}
\value{
Returns an object
of class \code{mc} which is a list with the following elements:
\describe{
\item{call}{Function call.}
\item{args}{Function arguments.}
\item{lm_process}{Processed \code{lm} object.}
\item{scale}{Sampling variance-covariance matrix of parameter estimates.}
\item{location}{Parameter estimates.}
\item{thetahatstar}{Sampling distribution of parameter estimates.}
\item{fun}{Function used ("MCMI").}
}
}
\description{
Generate the Sampling Distribution of Regression Parameters
Using the Monte Carlo Method for Data with Missing Values
}
\details{
Multiple imputation
is used to deal with missing values in a data set.
The vector of parameter estimates
and the corresponding sampling covariance matrix
are estimated for each of the imputed data sets.
Results are combined to arrive at the pooled vector of parameter estimates
and the corresponding sampling covariance matrix.
The pooled estimates are then used to generate the sampling distribution
of regression parameters.
See \code{MC()} for more details on the Monte Carlo method.
}
\examples{
# Data ---------------------------------------------------------------------
data("nas1982", package = "betaMC")
nas1982_missing <- mice::ampute(nas1982)$amp # data set with missing values

# Multiple Imputation
mi <- mice::mice(nas1982_missing, m = 5, seed = 42, print = FALSE)

# Fit Model in lm ----------------------------------------------------------
## Note that this does not deal with missing values.
## The fitted model (`object`) is updated with each imputed data
## within the `MCMI()` function.
object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982_missing)

# Monte Carlo --------------------------------------------------------------
mc <- MCMI(
  object,
  mi = mi,
  R = 100, # use a large value e.g., 20000L for actual research
  seed = 0508
)
mc
# The `mc` object can be passed as the first argument
# to the following functions
#   - BetaMC
#   - DeltaRSqMC
#   - DiffBetaMC
#   - PCorMC
#   - RSqMC
#   - SCorMC

}
\references{
Dudgeon, P. (2017).
Some improvements in confidence intervals
for standardized regression coefficients.
\emph{Psychometrika}, \emph{82}(4), 928–951.
\doi{10.1007/s11336-017-9563-z}

MacKinnon, D. P., Lockwood, C. M., & Williams, J. (2004).
Confidence limits for the indirect effect:
Distribution of the product and resampling methods.
\emph{Multivariate Behavioral Research}, \emph{39}(1), 99-128.
\doi{10.1207/s15327906mbr3901_4}

Pesigan, I. J. A., & Cheung, S. F. (2024).
Monte Carlo confidence intervals for the indirect effect with missing data.
\emph{Behavior Research Methods}.
\doi{10.3758/s13428-023-02114-4}

Preacher, K. J., & Selig, J. P. (2012).
Advantages of Monte Carlo confidence intervals for indirect effects.
\emph{Communication Methods and Measures}, \emph{6}(2), 77–98.
\doi{10.1080/19312458.2012.679848}
}
\seealso{
Other Beta Monte Carlo Functions: 
\code{\link{BetaMC}()},
\code{\link{DeltaRSqMC}()},
\code{\link{DiffBetaMC}()},
\code{\link{MC}()},
\code{\link{PCorMC}()},
\code{\link{RSqMC}()},
\code{\link{SCorMC}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Beta Monte Carlo Functions}
\keyword{betaMC}
\keyword{mc}
