#' Create a complete variability plot for spatial interaction models in a data
#' frame
#'
#' This function combines spatial variability interaction model representations
#' similar to the ones produced by [autoplot.sim_list()] into a single ggplot.
#' It provides an alternative graphical representation to the one produced by
#' [autoplot.sim_df()] and by [grid_autoplot()] for collection of spatial
#' interaction models in a `sim_df` object.
#'
#' The rationale of [autoplot.sim_df()] is to display a single value for each
#' spatial interaction model (SIM) in the `sim_df` data frame. On the contrary,
#' this function produces a graphical representation of the variability of a
#' partition of the SIMs in the data frame, using [autoplot.sim_list()] as the
#' graphical engine.
#'
#' The `key` parameter is used to partition the collection of SIMs. It can be
#' any expression which can be evaluated in the context of the `sim_df`
#' parameter. The function uses this parameter as the wrapping variable in a
#' call to [ggplot2::facet_wrap()]. It also uses it as a way to specific a
#' partition of the SIMs: each panel of the final figure is essentially the
#' variability graph generated by [autoplot.sim_list()] for the subset of the
#' SIMs in `sim_df` that match the corresponding value of `key`.
#'
#' Parameters of [ggplot2::facet_wrap()] can be set using the `fw_params`
#' parameter (in a list).
#'
#' @param sim_df a data frame of spatial interaction models, an object of class
#'   `sim_df`
#' @param key the wrapping variable which acts as group identifier for spatial
#'   interaction models
#' @inheritParams autoplot.sim_list
#' @param fw_params parameters for the [ggplot2::facet_wrap] call (if non
#'   `NULL`)
#' @param ... additional parameters passed to [autoplot.sim_list()]
#'
#' @returns a ggplot object
#' @export
#'
#' @examplesIf requireNamespace("ggplot2", quietly = TRUE)
#' positions <- as.matrix(french_cities[1:10, c("th_longitude", "th_latitude")])
#' distances <- french_cities_distances[1:10, 1:10] / 1000 ## convert to km
#' production <- rep(1, 10)
#' attractiveness <- log(french_cities$area[1:10])
#' all_flows <- grid_blvim(distances, production, seq(1.05, 1.45, by = 0.1),
#'   seq(1, 3, by = 0.5) / 400,
#'   attractiveness,
#'   bipartite = FALSE,
#'   epsilon = 0.1, iter_max = 1000,
#'   destination_data = list(
#'     names = french_cities$name[1:10],
#'     positions = positions
#'   ),
#'   origin_data = list(
#'     names = french_cities$name[1:10],
#'     positions = positions
#'   )
#' )
#' all_flows_df <- sim_df(all_flows)
#' ## group models by iteration number
#' grid_var_autoplot(all_flows_df, iterations)
#' ## or by convergence status (showing destination)
#' grid_var_autoplot(all_flows_df, converged,
#'   flow = "destination",
#'   with_names = TRUE
#' ) + ggplot2::coord_flip()
#' ## using positions
#' grid_var_autoplot(all_flows_df, iterations,
#'   flow = "destination",
#'   with_positions = TRUE
#' ) +
#'   ggplot2::scale_size_continuous(range = c(0, 3)) +
#'   ggplot2::coord_sf(crs = "epsg:4326")
grid_var_autoplot <- function(sim_df,
                              key,
                              flows = c(
                                "full", "destination",
                                "attractiveness"
                              ),
                              with_names = FALSE,
                              with_positions = FALSE,
                              cut_off = 100 * .Machine$double.eps^0.5,
                              adjust_limits = FALSE,
                              with_labels = FALSE,
                              qmin = 0.05,
                              qmax = 0.95,
                              normalisation = c("origin", "full", "none"),
                              fw_params = NULL,
                              ...) {
  rlang::check_installed("ggplot2", reason = "to use `grid_var_autoplot()`")
  with_cut_off <- !missing(cut_off)
  with_adjust_limits <- !missing(adjust_limits)
  with_with_labels <- !missing(with_labels)
  with_normalisation <- !missing(normalisation)
  if (!inherits(sim_df, "sim_df")) {
    cli::cli_abort("{.arg sim_df} must be a {.cls sim_df} object")
  }
  rlang::check_required(key)
  flows <- rlang::arg_match(flows)
  normalisation <- rlang::arg_match(normalisation)
  if (with_positions) {
    if (flows == "destination" || flows == "attractiveness") {
      if (is.null(destination_positions(sim_column(sim_df)))) {
        cli::cli_abort("Missing destination location positions")
      }
    }
    if (flows == "full") {
      cli::cli_warn(c("{.arg flows} = {.str full} cannot be combined with
{.arg with_positions} = {.val {TRUE}}",
        "!" = "proceeding with {.arg with_positions} set to {.val {FALSE}}"
      ))
      with_positions <- FALSE
    }
  }
  sim_list_autoplot_warning(
    flows,
    with_names, with_positions, with_cut_off, cut_off,
    with_adjust_limits, adjust_limits, with_with_labels,
    with_labels, with_normalisation, normalisation
  )
  expr <- rlang::enquo(key)
  val <- rlang::eval_tidy({{ expr }}, sim_df)
  val_name <- rlang::as_label(expr)
  flow_name <- ifelse(flows == "attractiveness", "attractiveness", "flow")
  col_names <- c("min", "Q_min", flow_name, "Q_max", "max")
  probs <- c(0, qmin, 0.5, qmax, 1)
  if (flows == "full") {
    the_cols <- 1:2
  } else {
    the_cols <- 1
  }
  if (with_normalisation && flows == "full") {
    pre_data <- tapply(
      sim_column(sim_df),
      val,
      function(a_sim) {
        sim_data <- fortify.sim_list(a_sim,
          data = NULL, flows = flows,
          with_names = FALSE, normalisation = normalisation
        )
        sim_data_stat <- quantile_sim_data(sim_data, flows, probs = probs)
        names(sim_data_stat) <- c(
          names(sim_data_stat)[the_cols], col_names
        )
        sim_data_stat
      }
    )
  } else {
    pre_data <- tapply(
      sim_column(sim_df),
      val,
      function(a_sim) {
        sim_data <- fortify.sim_list(a_sim,
          data = NULL, flows = flows,
          with_names = FALSE
        )
        sim_data_stat <- quantile_sim_data(sim_data, flows, probs = probs)
        names(sim_data_stat) <- c(
          names(sim_data_stat)[the_cols], col_names
        )
        sim_data_stat
      }
    )
  }
  if (is.factor(val)) {
    final_data <- combine_df(
      pre_data, factor(levels(val), levels(val)),
      val_name
    )
  } else {
    final_data <- combine_df(pre_data, sort(unique(val)), val_name)
  }
  pre <- sim_list_autoplot(
    sim_column(sim_df),
    final_data,
    flows,
    with_names,
    with_positions,
    cut_off,
    adjust_limits,
    with_labels,
    normalisation,
    ...
  )
  fw_parameters <- list(facets = ggplot2::vars(.data[[val_name]]))
  if (!is.null(fw_params)) {
    fw_parameters <- c(fw_parameters, fw_params)
  }
  pre +
    do.call(ggplot2::facet_wrap, fw_parameters)
}
