% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sequences.R
\name{sequences}
\alias{sequences}
\alias{sequence_}
\alias{seq_id}
\alias{seq_}
\alias{seq_size}
\alias{seq_start}
\alias{seq_end}
\alias{seq_increment}
\alias{window_sequence}
\alias{lag_sequence}
\alias{lead_sequence}
\title{Utilities for creating many sequences}
\usage{
sequence_(size, from = 1L, by = 1L, add_id = FALSE, as_list = FALSE)

seq_id(size)

seq_(
  from = NULL,
  to = NULL,
  by = NULL,
  size = NULL,
  add_id = FALSE,
  as_list = FALSE
)

seq_size(from, to, by = 1L)

seq_start(size, to, by = 1L)

seq_end(size, from, by = 1L)

seq_increment(size, from, to)

window_sequence(size, k, partial = TRUE, ascending = TRUE, add_id = FALSE)

lag_sequence(size, k, partial = TRUE, add_id = FALSE)

lead_sequence(size, k, partial = TRUE, add_id = FALSE)
}
\arguments{
\item{size}{Vector of sequence lengths.}

\item{from}{Start of sequence(s).}

\item{by}{Unit increment of sequence(s).}

\item{add_id}{Should the ID numbers of the sequences be added as names?
Default is \code{FALSE}.}

\item{as_list}{Should a list of sequences be returned?
Setting to \code{TRUE} would
place each distinct sequence vector into a distinct list element.
The default is \code{FALSE}.}

\item{to}{End of sequence(s).}

\item{k}{Window/lag size.}

\item{partial}{Should partial windows/lags be returned? Default is \code{TRUE}.}

\item{ascending}{Should window sequence be ascending? Default is \code{TRUE}.}
}
\value{
A vector of length \code{sum(size)} except for \code{seq_} which
returns a vector of size \code{sum((to - from) / (by + 1))}
}
\description{
\code{seq_} is a vectorised version of \link{seq} with some additional features. \cr
\code{seq_size} returns sequence sizes. \cr
\code{seq_start} returns sequence start points. \cr
\code{seq_end} returns sequence end points. \cr
\code{seq_increment} returns sequence increments. \cr
\code{sequence_} is an extension to \link{sequence} which
accepts decimal number increments. \cr
\code{seq_id} can be paired with \code{sequence_} to group individual sequences. \cr
\code{window_sequence} creates a vector of window sizes for rolling calculations. \cr
\code{lag_sequence} creates a vector of lags for rolling calculations. \cr
\code{lead_sequence} creates a vector of leads for rolling calculations. \cr
}
\details{
\code{seq_()} is a fast vectorised version of \code{seq()} with powerful features.
It can return many sequences as a single vector of combined sequences
or a list of sequences.

\code{sequence_()} works in the same way as \code{sequence()} but can accept
non-integer \code{by} values.
This is the workhorse function of \code{seq_()}.

Unlike \code{sequence()}, \code{sequence_()} recycles all its arguments,
including \code{size}.

If any of the sequences contain values > \code{.Machine$integer.max},
then the result will always be a double vector.
}
\examples{
library(cheapr)

# These two functions are similar
sequence(1:3);sequence_(1:3)

# sequence_() can handle any numeric vector sequence
sequence(1:3, by = 0.1);sequence_(1:3, by = 0.1)

# Alternatively return as a list of sequences

sequence_(1:3, by = 0.1, as_list = TRUE)

# Add IDs to the sequences
sequence_(1:3, by = 0.1, add_id = TRUE)
# Turn this quickly into a data frame
seqs <- sequence_(1:3, by = 0.1, add_id = TRUE)
new_df(name = names(seqs), seq = seqs)

sequence(c(3, 2), by = c(-0.1, 0.1));sequence_(c(3, 2), by = c(-0.1, 0.1))

# Vectorised version of seq()
seq_(1, 10, by = c(1, 0.5))
# Same as above
c(seq(1, 10, 1), seq(1, 10, 0.5))

# Again, as a list of sequences
# 2 different start points and 2 different increments
seq_(from = c(-1, 1), 3, by = c(1, 0.5), as_list = TRUE)

# Programmers may use seq_size() to determine final sequence lengths

sizes <- seq_size(1, 10, by = c(1, 0.5))
print(paste(c("sequence sizes: (", sizes, ") total size:", sum(sizes)),
            collapse = " "))

# Or return as a list of sequences
# Note that these lengths will match the above line of code
seq_(1, 10, by = c(1, 0.5), as_list = TRUE) |>
  list_lengths()

# Sequences of dates with different increments

from <- Sys.Date()
to <- from + 10
by <- c(1, 2, 3)
date_seqs <- seq_(from, to, by, as_list = TRUE)
lapply(date_seqs, function(x) `class<-`(x, "Date"))

# Utilities for rolling calculations

# A window sequence of size 3 for a vector of size 10
# This tells us how big the window should be when looking backwards
window_sequence(10, 3, partial = FALSE)
window_sequence(10, 3, partial = TRUE)

window_sequence(c(3, 5), 3)
window_sequence(c(3, 5), 3, partial = FALSE)
window_sequence(c(3, 5), 3, partial = TRUE, ascending = FALSE)

# Lag sequence of size 3 for a vector of size 10
# This tells us how for we should look backwards at any given point
lag_sequence(10, 3, partial = FALSE)
# How far to look forwards
lead_sequence(10, 3, partial = FALSE)

lag_sequence(10, 3, partial = TRUE)
lead_sequence(10, 3, partial = TRUE)
# One can for example use these in data.table::frollsum

}
