#' Sanitize CLI message
#'
#' Escape braces in error messages so that \pkg{cli} does not treat them
#' as inline formatting.
#'
#' @param res A character vector of error messages.
#'
#' @returns A character vector with braces escaped for use in `cli` messages.
#'
#' @examples
#' sanitize_cli("Value {x} is invalid")
#'
#' @importFrom stringr str_replace_all str_escape
#' @export
sanitize_cli <- function(res) {
  if (isTRUE(res)) {
    return(res)
  } else {
    res |>
      stringr::str_replace_all(stringr::str_escape("{"), "{{") |>
      stringr::str_replace_all(stringr::str_escape("}"), "}}")
  }
}

#' Format CLI bullets
#'
#' Convert error messages into a named vector suitable for `cli::cli_abort()`.
#'
#' @param res A character vector of messages (typically from a `check_*` call).
#' @param cli_bullet Single character bullet type (`"i"`, `"x"`, etc.).
#'
#' @returns A named character vector where names are bullet types.
#'
#' @examples
#' fmt_bullet_cli("Something went wrong")
#'
#' @importFrom stats setNames
#' @export
fmt_bullet_cli <- function(res, cli_bullet = "i") {
  if (isTRUE(res)) {
    return(res)
  } else {
    res <- res |>
      sanitize_cli() |>
      stats::setNames(rep(cli_bullet, length(res)))
    res
  }
}

#' Fix braced lists in messages
#'
#' Rewrite brace-enclosed lists such as \code{{'a','b','c'}} into \code{{a}, {b}, {c}}
#' to improve how they are rendered by \pkg{cli}.
#'
#' @param msg A single character string with a message.
#'
#' @returns A modified message string.
#'
#' @examples
#' fix_braced_list("allowed values: {'a','b','c'}")
#'
#' @export
fix_braced_list <- function(msg) {
  # Find first {...} block
  m <- regexpr("\\{[^}]*\\}", msg)
  if (m[1] == -1L) return(msg)

  block <- regmatches(msg, m)[[1]]      # e.g. "{'a','b','c'}"

  # Extract inside braces
  inner <- sub("^\\{(.*)\\}$", "\\1", block)  # "'a','b','c'"

  # If no comma inside the braces, do nothing
  if (!grepl(",", inner, fixed = TRUE)) {
    return(msg)
  }

  # Split on commas and trim
  items <- strsplit(inner, ",", fixed = TRUE)[[1]]
  items <- trimws(items)

  # Wrap each item in its own braces
  new_block <- paste0("{", items, "}", collapse = ", ")

  # Replace in original message
  regmatches(msg, m) <- list(new_block)
  msg
}

#' Make CLI-style assertion
#'
#' Internal helper used by all `assert_*_cli()` functions.
#'
#' @param x The object being checked.
#' @param res Result of a `checkmate::check_*()` call (`TRUE` or message).
#' @param var.name Name of the variable for error messages.
#' @param collection Optional [checkmate::AssertCollection] to collect failures.
#'
#' @returns Invisibly returns `x` on success or raises a `cli::cli_abort()` error.
#'
#' @examples
#' # Typically used via higher-level wrappers:
#' make_assertion(1L, checkmate::check_int(1L), "x", NULL)
#'
#' @importFrom glue glue
#' @export
make_assertion <- function(x, res, var.name, collection) {
  if (!isTRUE(res)) {
    checkmate::assertString(var.name, .var.name = ".var.name")

    if (is.null(collection)) {
      res <- fix_braced_list(res)
      cli::cli_abort(
        c(
          "x" = "Assertion on { var.name } failed.",
          "i" = res
        ),
        #.internal = TRUE,
        call = parent.frame(n=2)
      )
    }
    checkmate::assertClass(collection, "AssertCollection", .var.name = "add")
    collection$push(
      glue::glue("Variable '{ var.name }': {res}")
    )
  }
  return(invisible(res))
}

#' Combine multiple CLI assertions
#'
#' Combine multiple `check_*_cli()` expressions with `"or"` or `"and"`
#' logic and assert them jointly.
#'
#' @param ... Expressions evaluating to the result of `check_*_cli()` calls.
#' @param combine Either `"or"` or `"and"`.
#' @param .var.name Optional variable name(s) for the combined assertion.
#' @param add Optional [checkmate::AssertCollection] to collect failures.
#'
#' @returns Invisible `TRUE` on success, otherwise error or collected failures.
#'
#' @examples
#' x <- 1L
#' assert_cli(
#'   check_int_cli(x),
#'   check_numeric_cli(x),
#'   combine = "or",
#'   .var.name = "x"
#' )
#'
#' @export
assert_cli <- function(..., combine = "or", .var.name = NULL, add = NULL) {
  checkmate::assertChoice(combine, c("or", "and"))
  checkmate::assertClass(add, "AssertCollection", .var.name = "add", null.ok = TRUE)
  dots = match.call(expand.dots = FALSE)$...
  checkmate::assertCharacter(.var.name, null.ok = TRUE, min.len = 1L, max.len = length(dots))
  env = parent.frame()
  if (combine == "or") {
    if (is.null(.var.name))
      .var.name = vapply(dots, function(x) as.character(x)[2L], FUN.VALUE = NA_character_)
    msgs = character(length(dots))
    for (i in seq_along(dots)) {
      val = eval(dots[[i]], envir = env)

      if (isTRUE(val))
        return(invisible(TRUE))
      msgs[i] = as.character(val)
    }
    .mstopOrPush(res = msgs, v_name = .var.name, collection = add)
  } else {
    for (i in seq_along(dots)) {
      val = eval(dots[[i]], envir = env)
      if (!isTRUE(val)) {
        if (is.null(.var.name))
          .var.name = as.character(dots[[i]])[2L]
        msgs = as.character(val)
        .mstopOrPush(res = msgs, v_name = .var.name, collection = add)
      }
    }
  }
  invisible(TRUE)
}

.mstopOrPush = function(res, v_name, collection = NULL) {
  if (!is.null(collection)) {
    purrr::map2(res, names(res), \(e, n) glue::glue("({n}): {e}")) |>
      as.character() |>
      collection$push()
  } else if (length(v_name) > 1L) {
    v_name <- v_name |>
      unique()
    names(res) <- rep("i", length(res))
    msg <- c(
      glue::glue("Assertion on < v_name > failed.", .open = "<", .close = ">"),
      res
    )
    cli::cli_abort(
      msg,
      #.internal = TRUE,
      call = parent.frame(n=2)
    )
  } else {
    msg <- c(
      glue::glue("Assertion on < v_name > failed.", .open = "<", .close = ">"),
      c("i" = res)
    )
    cli::cli_abort(
      msg,
      #.internal = TRUE,
      call = parent.frame(n=2)
    )
  }
}
