% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/numerical.mle.R
\docType{package}
\name{compositional.mle-package}
\alias{compositional.mle-package}
\alias{compositional.mle}
\title{compositional.mle: Compositional Maximum Likelihood Estimation}
\description{
Provides composable optimization strategies for maximum likelihood estimation (MLE). Solvers are first-class functions that combine via sequential chaining, parallel racing, and random restarts. Implements gradient ascent, Newton-Raphson, quasi-Newton (BFGS), and derivative-free methods with support for constrained optimization and tracing. Returns 'mle' objects compatible with 'algebraic.mle' for downstream analysis. Methods based on Nocedal J, Wright SJ (2006) "Numerical Optimization" \doi{10.1007/978-0-387-40065-5}.

A domain-specific language for maximum likelihood estimation where solvers
are first-class composable functions. Following SICP principles, solvers
combine via sequential chaining, parallel racing, and iteration to build
sophisticated optimization strategies from simple primitives.
}
\section{The Problem Abstraction}{

\code{\link{mle_problem}} encapsulates a statistical estimation problem:
\itemize{
  \item Log-likelihood function
  \item Optional analytic score and Fisher information (computed numerically if not provided)
  \item Domain constraints
  \item Metadata (parameter names, observation count)
}
}

\section{Solver Factories}{

Solver factories return solver functions with signature
\code{(problem, theta0, trace) -> mle_result}:
\itemize{
  \item \code{\link{gradient_ascent}}: First-order gradient method
  \item \code{\link{newton_raphson}}: Second-order Newton's method
  \item \code{\link{bfgs}}: Quasi-Newton BFGS
  \item \code{\link{nelder_mead}}: Derivative-free simplex
  \item \code{\link{grid_search}}: Exhaustive grid search
}
}

\section{Composition Operators}{

Combine solvers to build complex strategies:
\itemize{
  \item \code{\link{\%>>\%}}: Sequential chaining (coarse-to-fine)
  \item \code{\link{\%|\%}}: Parallel racing (try multiple, pick best)
  \item \code{\link{with_restarts}}: Multiple random starting points
}
}

\section{Tracing}{

\code{\link{mle_trace}} configures what to track during optimization
(values, path, gradients, timing) for diagnostics and visualization.
}

\examples{
\donttest{
# Define problem
set.seed(42)
x <- rnorm(100, mean = 5, sd = 2)
problem <- mle_problem(
  loglike = function(theta) sum(dnorm(x, theta[1], theta[2], log = TRUE)),
  constraint = mle_constraint(support = function(theta) theta[2] > 0,
                              project = function(theta) c(theta[1], max(theta[2], 1e-8)))
)

# Simple solve
result <- gradient_ascent()(problem, c(0, 1))

# Composed strategy: grid -> gradient -> Newton
strategy <- grid_search(lower = c(-10, 0.1), upper = c(10, 5), n = 5) \%>>\%
  gradient_ascent() \%>>\% newton_raphson()
result <- strategy(problem, c(0, 1))

# Race different methods
strategy <- gradient_ascent() \%|\% bfgs() \%|\% nelder_mead()
result <- strategy(problem, c(0, 1))
}

}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/queelius/compositional.mle}
  \item \url{https://queelius.github.io/compositional.mle/}
  \item Report bugs at \url{https://github.com/queelius/compositional.mle/issues}
}

}
\author{
\strong{Maintainer}: Alexander Towell \email{queelius@gmail.com} (\href{https://orcid.org/0000-0001-6443-9897}{ORCID})

}
\keyword{internal}
