% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dcf.R
\name{cf_make_full_table}
\alias{cf_make_full_table}
\title{Assemble the full cash-flow table (discounted cash flow and debt)}
\usage{
cf_make_full_table(dcf, schedule)
}
\arguments{
\item{dcf}{A \code{list} containing at least an element \code{cashflows} (data.frame or tibble)
with one row per \code{year} and the following columns:
\itemize{
\item \code{year} (integer, 0 = acquisition date),
\item \code{net_operating_income} (numeric),
\item \code{capex} (numeric, optional),
\item \code{free_cash_flow} (numeric, pre-debt cash flow),
\item \code{sale_proceeds} (numeric, sale proceeds in the exit year, 0 otherwise),
\item \code{discount_factor} (numeric, strictly positive discount factor).
}

If \code{gei} or \code{noi} are missing, they are derived according to the convention:
\code{gei := net_operating_income} and \code{noi := gei - opex}. If \code{opex} is missing,
it is set to 0.}

\item{schedule}{A data.frame or tibble of the debt schedule with one row per
\code{year} and the required columns:
\itemize{
\item \code{year} (integer, aligned with \code{dcf$cashflows$year}),
\item \code{debt_draw} (numeric, drawdown; typically positive at \code{year == 0}),
\item \code{interest} (numeric),
\item \code{amortization} (numeric),
\item \code{payment} (numeric, debt service = interest + amortization; must be 0 at \code{year == 0}),
\item \code{arrangement_fee} (numeric, upfront or recurring fees),
\item \code{outstanding_debt} (numeric, end-of-period outstanding balance).
}}
}
\value{
A merged tibble (join on \code{year}) containing:
\itemize{
\item all input columns from the  Discounted Cash Flow (DCF) and the debt schedule,
\item \code{df} (alias of \code{discount_factor}),
\item \code{cf_pre_debt} (= \code{free_cash_flow}),
\item \code{cf_post_debt} (= \code{free_cash_flow - payment - arrangement_fee + debt_draw}),
\item \code{equity_flow} (= \code{cf_post_debt + sale_proceeds}),
\item \code{equity_disc} (= \code{equity_flow / df}).
}
}
\description{
Builds an annual table by merging operating cash flows from a discounted cash
flow model with a debt schedule; standardises gross effective income (GEI) and
net operating income (NOI), computes post-debt cash flows, the equity cash flow,
and discounted equity cash flows. Enforces a minimal contract on expected
columns on both inputs.
}
\details{
Invariants and checks:
\itemize{
\item Stop if required columns are missing on the  Discounted Cash Flow (DCF) or the debt side.
\item Stop if \code{payment[year == 0] != 0}.
\item Warn if \code{debt_draw[year == 0] <= 0}.
}
}
\examples{
cf <- tibble::tibble(
  year = 0:2,
  net_operating_income = c(NA, 120, 124),
  opex = c(0, 20, 21),
  capex = c(0, 5, 5),
  free_cash_flow = c(-100, 95, 98),
  sale_proceeds = c(0, 0, 150),
  discount_factor = c(1, 1.05, 1.1025)
)
dcf <- list(cashflows = cf)

schedule <- tibble::tibble(
  year = 0:2,
  debt_draw = c(60, 0, 0),
  interest = c(0, 3, 2),
  amortization = c(0, 10, 50),
  payment = interest + amortization,
  arrangement_fee = c(0.6, 0, 0),
  outstanding_debt = c(60, 50, 0)
)

res <- cf_make_full_table(dcf, schedule)
res

}
